import React from "react";
import moment from "moment/moment";
import gql from "graphql-tag";
import { Mutation, graphql } from "react-apollo";
import PropTypes from "prop-types";
import Spinner from "react-spinkit";
import { Button, Glyphicon } from "react-bootstrap";

const createServiceAuditQuery = gql`
  mutation createServiceAudit($service_audit: ServiceAuditInput!) {
    service_audit: createServiceAudit(service_audit: $service_audit) {
      id
      audit_type
      auditor
      action
      audit_time
      audit_value
    }
  }
`;

class OwnerValidationButtons extends React.Component {
  render() {
    let {serviceID, auditValue, serviceAudits} = this.props;
    return <Mutation
      mutation={createServiceAuditQuery}
      update={(cache, {data: {service_audit}}) => {
        let newAudits = []
        let foundOwnerAudit = false;
        // If the Audit for that type already exists, replace it. Otherwise append if not found
        for(let i = 0; i < serviceAudits.length; i++) {
          let audit = serviceAudits[i];
          if (audit.audit_type === "owner") {
            foundOwnerAudit = true;
            newAudits.push(service_audit)
          } else {
            newAudits.push(audit)
          }
        }
        if (!foundOwnerAudit) {
          newAudits.push(service_audit);
        }
        cache.writeFragment({
          id: `Service:${serviceID}`,
          fragment: gql`
            fragment service_audits on Service {
              latest_service_audits {
                id
                audit_type
                auditor
                action
                audit_time
                audit_value
              }
              __typename
            }
          `,
          data: {
            latest_service_audits: newAudits,
            __typename: "Service"
          }
        })
      }}
    >
      {(createAudit, {loading, error}) => {
        let errorComponent = null;
        if (error) {
          errorComponent = <p><strong>We encountered error when recording your validation. Please retry. Error was {error.message}</strong></p>
        }
        if (loading) {
          return <Spinner name={"line-scale"} color={"purple"} noFadeIn/>
        }
        // Get the service list of service audits, then add the given service audit into it.
        return <div className={"pull-right"}>
          {errorComponent}
          <Button
            onClick={() => createAudit({
              variables: {
                service_audit: {
                  service_id: serviceID,
                  audit_type: "owner",
                  action: "validated",
                  audit_time: moment.utc().format(),
                  audit_value: auditValue,
                }
              }
            })}

            bsSize="xsmall" bsStyle="success"><Glyphicon glyph="ok"/></Button>
          &nbsp;
          <Button
            onClick={() => createAudit({
              variables: {
                service_audit: {
                  service_id: serviceID,
                  audit_type: "owner",
                  action: "invalidated",
                  audit_time: moment.utc().format(),
                  audit_value: auditValue,
                }
              }
            })}

            bsSize="xsmall" bsStyle="danger"><Glyphicon glyph="ban-circle"/></Button>
        </div>
      }
      }
    </Mutation>
  }
}

OwnerValidationButtons.propTypes = {
  serviceID: PropTypes.string,
  auditValue: PropTypes.string,
};

export default OwnerValidationButtons
