import React from "react";
import _ from "lodash";
import { getUser } from "lib/auth";
import PropTypes from "prop-types";
import { Alert, ToggleButton, Button, ToggleButtonGroup } from "react-bootstrap";
import { serviceShallowDataFragment } from "common/goracle/actions";
import {
  generateValidationFormatter,
  serviceFormatter, managerFormatter,
  teamFormatter, orgFormatter, buFormatter,
  tierFormatter,
} from "../../../lib/formatters";
import OwnerValidationButtons from "./ServiceAudit";
import PrimaryOwnerWrapper from "./ServiceOwnerPrimaryView";
import ServiceSlackChannel from "./ServiceSlackChannel";
import AvailabilityTable from "../../../common/ServiceTable/containers/ServiceContainer";
import { getUserDisplayName } from '../../../common/User/Util';
import ServiceState from "./ServiceState";
import WrapperTooltip from "../../../components/WrapperTooltip";
import { READ_ONLY_MODE } from "../../../constants";

var editEnabled = false;

export function isEditEnabled() {
  return editEnabled;
}

function primaryOwnerFormatter(s) {
  let po = _.get(s, "primary_owner", null);
  if (po !== null) {
    return {
      display: <PrimaryOwnerWrapper service={s} isediting={false} editEnabled={editEnabled} />,
      value: getUserDisplayName(po)
    }
  } else {
    return {
      display: "Unknown",
      value: "Unknown",
    };
  }
}

function stateFormatter(service) {
  if (!service) {
    return {
      display: '',
      value: '',
    }
  } else {
    return {
      display: <ServiceState serviceID={service.id} isEditing={false} serviceState={service.state} />,
      value: service.state ? service.state : 'unknown',
    }
  }
}

function serviceOwnerValidationCreator(s)  {
  return <OwnerValidationButtons serviceID={s.id} auditValue={s.primary_owner.preferred_name} serviceAudits={s.latest_service_audits}/>
}

function slackFormatter(service) {
  if (!service) {
    return {
      display: '',
      value: '',
    }
  } else {
    return {
      display: <ServiceSlackChannel serviceID={service.id} isEditing={false} slackChannel={service.slack_channel} />,
      value: service.slack_channel ? service.slack_channel.name : '',
    }
  }
}

export var serviceOwnerColumns = [
  {
    title: "Name",
    key: "name",
    filter: "text",
    isBasicFilter: true,
    formatter: serviceFormatter,
  },
  {
    title: "Primary Owner",
    key: "primary_owner.cn",
    formatter: primaryOwnerFormatter,
    filter: "select",
    isBasicFilter: true,
  },
  {
    title: "Service State",
    key: "state",
    formatter: stateFormatter,
    filter: "select",
    isBasicFilter: true,
  },
  {
    title: "Tier",
    key: "tier", // doesnt matter
    formatter: tierFormatter,
    filter: "select",
    isBasicFilter: true,
  },
  {
    title: "Manager",
    key: "primary_owner.manager",
    formatter: managerFormatter,
    filter: "select",
    isBasicFilter: false,
  },
  {
    title: "Team",
    filter: "select",
    isBasicFilter: false,
    key: "team_name",
    formatter: teamFormatter,
  },
  {
    title: "Org",
    filter: "select",
    isBasicFilter: false,
    key: "org_name",
    formatter: orgFormatter,
  },
  {
    title: "Business Unit",
    filter: "select",
    isBasicFilter: false,
    key: "bu_name",
    formatter: buFormatter,
  },
  {
    title: "Slack Channel",
    key: 'slack_channel',
    formatter: slackFormatter,
  },
  {
    title: "Primary Owner Validation",
    key: "latest_primary_owner_audit",
    formatter: generateValidationFormatter("owner", true, true, true, serviceOwnerValidationCreator),
    filter: "select",
    isBasicFilter: true,
  },
];

class ServiceOwnerView extends React.Component {
  constructor(props, context) {
    super(props, context);
    this.handleChange = this.handleChange.bind(this);
    this.state = {
      isEditEnabled: 0
    };
  }

  handleChange(e) {
    if (e === 1) {
      editEnabled = true
    } else {
      editEnabled = false
    }
    this.setState({ isEditEnabled: e });
  }

  render()
  {
    let { services, uid } = this.props;
    return <div>
      <h2> Services owned by {uid}
        <ToggleButtonGroup
          className="pull-right"
          type="radio"
          name="edit-options"
          defaultValue={0}
          onChange={this.handleChange}
        >
          <ToggleButton style={{color: 'purple'}} value={0}><WrapperTooltip title={"View Mode"}/></ToggleButton>
          {READ_ONLY_MODE ? (
            <Button disabled>Edit Mode</Button>
            ) : (
            <ToggleButton style={{ color: "white", backgroundColor: "#f0ad4e" }} value={1}><WrapperTooltip title={"Edit Mode"} /></ToggleButton>
          )}
        </ToggleButtonGroup>
      </h2>
      <AvailabilityTable
        tableId="servicesOwnerIndex"
        data={services}
        dataType="service"
        columns={serviceOwnerColumns}
        enableFilters={true}
        showExports={true}
        defaultItemsPerPage={50}
        isEditEnabled={this.state.isEditEnabled}
      />
    </div>;
  }
}

ServiceOwnerView.propTypes = {
  services: PropTypes.array,
  uid: PropTypes.string,
}

export default ServiceOwnerView;
