import React from "react";
import PropTypes from "prop-types";
import _ from "lodash";
import {
  Modal,
  Table,
  ButtonToolbar,
  Button,
  Glyphicon,
  Panel,
  Row,
  Col,
} from "react-bootstrap";
import { Link } from "react-router-dom";
import moment from "moment";
import WrapperTooltip from "components/WrapperTooltip";

const auditTypesToLabels = {
  component: "Component",
  latency: "Latency",
  owner: "Primary Owner",
  dependencies: "Dependencies",
  state: "Service State",
};

// Looks for the service audit of the given type
// among the list of lastest service audits
// If one is not found, returns the string "N/A"
function formatLastAuditMessage(audits, auditType) {
  if (!audits) {
    return [];
  }
  var audit = null;
  for (var i = 0; i < audits.length; i++) {
    if (audits[i].audit_type === auditType) {
      var audit = audits[i];
    }
  }
  if (audit === null) {
    return "N/A";
  }
  var timeString = moment(audit.audit_time).format("YYYY-MM-DD HH:mm:ss");
  return `${audit.action} ${audit.audit_value} by ${audit.auditor} at ${timeString}`;
}

function getAuditsOfType(type, audits) {
  if (type === "all") {
    return audits;
  }
  var auditsOfType = [];
  if (audits) {
    for (var i = 0; i < audits.length; i++) {
      if (audits[i].audit_type === type) {
        auditsOfType.push(audits[i]);
      }
    }
  }
  return auditsOfType;
}

class ServiceAudits extends React.Component {
  render() {
    const {
      service,
      createServiceAudit,
      toggleModal,
      modalState,
      isEditing,
    } = this.props;

    if (isEditing || service === undefined) {
      return <span />;
    }

    let latest_audits = service.latest_service_audits;
    var createAudit = (type, action) => {
      if (type === "owner") {
        createServiceAudit(service.id, type, action, service.primary_owner.preferred_name);
      } else if (type === "state") {
        createServiceAudit(service.id, type, action, service.state);
      } else {
        createServiceAudit(service.id, type, action);
      }
    }
    var showModal = type => toggleModal(true, type);
    var hideModal = type => toggleModal(false, type);
    return (
      <Panel header="Service Validations">
        <Table responsive hover bordered>
          <thead>
            <tr>
              <th>
                <WrapperTooltip
                  title="Actions"
                  tooltipID="service_validation_actions"
                />
              </th>
              <th>
                <WrapperTooltip
                  title="Validation Type"
                  tooltipID="service_validation_type"
                />
              </th>
              <th>
                <WrapperTooltip
                  title="Last Validation"
                  tooltipID="service_validation_last"
                />
              </th>
              <th>
                <WrapperTooltip
                  title="History"
                  tooltipID="service_validation_history"
                />
              </th>
            </tr>
          </thead>
          <tbody>
            {_.keys(auditTypesToLabels).map(function(auditType) {
              return ((auditType === "dependencies" || auditType === "owner" || auditType === "state") &&
                <tr key={auditType}>
                  <td>
                    <ButtonToolbar>
                      <Button
                        onClick={() => createAudit(auditType, "validated")}
                        bsSize="xsmall"
                        bsStyle="success"
                      >
                        <Glyphicon glyph="ok" />
                      </Button>
                      <Button
                        onClick={() => createAudit(auditType, "invalidated")}
                        bsSize="xsmall"
                        bsStyle="danger"
                      >
                        <Glyphicon glyph="ban-circle" />
                      </Button>
                    </ButtonToolbar>
                  </td>
                  <td>{auditTypesToLabels[auditType]}</td>
                  <td>{formatLastAuditMessage(latest_audits, auditType)}</td>
                  <td>
                    <Button bsSize="small" onClick={() => showModal(auditType)}>
                      Show
                    </Button>
                  </td>
                </tr>
              );
            })}
          </tbody>
        </Table>
        <Button bsSize="small" onClick={() => showModal("all")}>
          Full Audit Log
        </Button>
        <Modal show={modalState.enabled} onHide={hideModal}>
          <Modal.Header closeButton>
            <Modal.Title>
              Service Audits: {auditTypesToLabels[modalState.auditType]}
            </Modal.Title>
          </Modal.Header>
          <Modal.Body>
            <Table responsive hover bordered>
              <thead>
                <tr>
                  <th>Audit Type</th>
                  <th>Auditor</th>
                  <th>Time</th>
                  <th>Action</th>
                  <th>Audit Value</th>
                </tr>
              </thead>
              <tbody>
                {getAuditsOfType(
                  modalState.auditType,
                  service.service_audits
                ).map(function(sa) {
                  return (
                    <tr key={sa.id}>
                      <td>{auditTypesToLabels[sa.audit_type]}</td>
                      <td>{sa.auditor}</td>
                      <td>
                        {moment(sa.audit_time).format("YYYY-MM-DD HH:mm:ss")}
                      </td>
                      <td>{sa.action}</td>
                      <td>{sa.audit_value}</td>
                    </tr>
                  );
                })}
              </tbody>
            </Table>
          </Modal.Body>
        </Modal>
      </Panel>
    );
  }
}

ServiceAudits.propTypes = {
  service: PropTypes.object.isRequired,
  createServiceAudit: PropTypes.func.isRequired,
  toggleModal: PropTypes.func.isRequired,
  modalState: PropTypes.object.isRequired,
  isEditing: PropTypes.bool,
};

export default ServiceAudits;
