import React from "react";
import PropTypes from "prop-types";
import { Panel, Col, Row } from "react-bootstrap";
import MessagingButtons from "components/MessagingButtons";
import { FormMultiSelectField } from "common/FormFields";
import { dependencyServiceColumns, serviceFormatter } from "lib/formatters";
import { WrapperTooltip } from "components/WrapperTooltip";
import ServiceDependencyTree from "components/ServiceDependencyTree";
import gql from "graphql-tag";
import _ from "lodash";
import { displayDownstreamSlackLink, displayUpstreamSlackLink } from "../containers/ServiceViewContainer";

function upstreamNormalizer(downID) {
  return function(val, prevVal) {
    var { action, object, id } = val;
    // Create a new object for state
    var newVal = [];
    let found = false;
    // Add all the old values, excluding the
    // one we are removing, if we are indeed removing
    for (var i = 0; i < prevVal.length; i++) {
      if (prevVal[i].root_service.id === id) {
        found = true;
        if (action === "remove") {
          continue;
        }
      }
      newVal.push(prevVal[i]);
    }
    // If adding, add now
    if (action === "add" && !found) {
      // Push a service dependency
      // FIXME: not sure how I feel about the typename stuff, this seems to be
      // related to a UI status in the availability table
      newVal.push({
        downstream_service: {
          id: downID,
          __typename: "Service",
        },
        root_service: { id: id, name: object.name, __typename: "Service" },
        __typename: "ServiceDependency",
      });
    }

    return newVal;
  };
}

function downstreamNormalizer(rootID) {
  return function(val, prevVal) {
    var { action, object, id, rollupKey } = val;
    // Create a new object for state
    var newVal = [];
    let found = false;
    // Add all the old values, excluding the
    // one we are removing, if we are indeed removing
    for (var i = 0; i < prevVal.length; i++) {
      if (prevVal[i].downstream_service.id === id) {
        found = true;
        if (action === "remove") {
          continue;
        }
      }
      newVal.push(prevVal[i]);
    }
    // If adding, add now
    if (action === "add" && !found) {
      // Push a service dependency
      // FIXME: not sure how I feel about the typename stuff,
      // this seems to be related to a UI status in the availability table
      newVal.push({
        root_service: {
          id: rootID,
          __typename: "Service",
        },
        downstream_service: {
          id: id,
          name: object.name,
          __typename: "Service",
        },
        __typename: "ServiceDependency",
      });
    }

    return newVal;
  };
}

const upstreamService = d => {
  return d.root_service;
};

const downstreamService = d => {
  return d.downstream_service;
};

class ServiceDependencies extends React.Component {
  render() {
    const {
      serviceID,
      service,
      isEditing,
      upstreamSlacks,
      downstreamSlacks,
    } = this.props;
    let optionsQuery = gql`
      query Services {
        services {
          id
          name
        }
      }
    `;
    let optionsQueryVars = {};
    var upstreamSlackList, downstreamSlackList;
    // keep upstreamSlackList empty until all upstream service slacks have been fetched
    if (!displayUpstreamSlackLink()) {
      upstreamSlackList = []
    } else {
      upstreamSlackList = upstreamSlacks
    }
    // keep downstreamSlackList empty until all downstream service slacks have been fetched
    if (!displayDownstreamSlackLink()) {
      downstreamSlackList = []
    } else {
      downstreamSlackList = downstreamSlacks
    }
    return (
      <div>
        <Panel header={<WrapperTooltip title="Dependencies" />}>
          <Row>
            <Col xs={8} md={8}>
              <h4>Upstream Services (Services that depend on, or call the {service && `"${service.name}"`} service)</h4>
            </Col>
            <Col xs={4} md={4}>
              {!isEditing && (
                <div className={'pull-right'}>
                  <MessagingButtons
                    titleFunc={type => {
                      return type + " Service Upstreams";
                    }}
                    contacts={upstreamSlackList}
                    defaultReplyTo={_.get(service.team, "email", "")}
                  />
                  <i>(notifies valid slack-channels of all upstream services)</i>
                </div>
              )}
            </Col>
          </Row>
          <FormMultiSelectField
            name="service_upstreams"
            id="serviceUpstreamsEdit"
            dataType="service"
            dataFunc={upstreamService}
            columns={dependencyServiceColumns}
            normalizer={upstreamNormalizer(serviceID)}
            optionNameFormatter={serviceFormatter}
            optionsQuery={optionsQuery}
            optionsQueryVars={optionsQueryVars}
            isEditing={isEditing}
          />
          <Row>
            <Col xs={8} md={8}>
              <h4>Downstream Services (Services that this service {service && `"${service.name}"`} depends on or calls)</h4>
            </Col>
            <Col xs={4} md={4}>
              {!isEditing && (
                <div className={'pull-right'}>
                  <MessagingButtons
                    titleFunc={type => {
                      return type + " Service Downstreams";
                    }}
                    contacts={downstreamSlackList}
                    defaultReplyTo={_.get(service.team, "email", "")}
                  />
                  <i>(notifies valid slack-channels of all downstream services)</i>
                </div>
              )}
            </Col>
          </Row>
          <FormMultiSelectField
            name="service_downstreams"
            id="serviceDownstreamsEdit"
            dataType="service"
            dataFunc={downstreamService}
            columns={dependencyServiceColumns}
            normalizer={downstreamNormalizer(serviceID)}
            optionNameFormatter={serviceFormatter}
            optionsQuery={optionsQuery}
            optionsQueryVars={optionsQueryVars}
            isEditing={isEditing}
          />
          {!isEditing && (
            <div>
              <h4>Dependency Tree</h4>
              <ServiceDependencyTree serviceID={serviceID}/>
            </div>
          )}
        </Panel>
      </div>
    );
  }
}

ServiceDependencies.propTypes = {
  isEditing: PropTypes.bool,
  serviceID: PropTypes.string.isRequired,
};

export default ServiceDependencies;
