// @format
import React from "react";
import PropTypes from "prop-types";
import _ from "lodash";
import { Col, ListGroup, ListGroupItem, Panel, Row } from "react-bootstrap";
import { toSlack } from "lib/formatters";
import { Field } from "redux-form";
import FormText from "common/FormFields/components/FormText";
import FormSelect from "common/FormFields/containers/FormSelect";
import FormTextArea from "common/FormFields/components/FormTextArea";
import FormSelectType from "common/FormFields/components/FormSelectType";
import { FormAttributeSelectField } from "common/FormFields";
import UserSelect from "common/User/UserSelect";
import gql from "graphql-tag";
import ServiceDetailHierarchy from "../../ServiceOwners/components/ServiceDetailHierarchy";
import { serviceStates } from "../../../routes/ServiceOwners/components/ServiceStateDropdown";

const slackChannelsQuery = gql`
  query slackChannels {
    slack {
      channels {
        name
        id
      }
    }
  }
`;

const serviceOptions = [
  { value: "external-product", label: "External Product" },
  { value: "internal-product", label: "Internal Product" },
  {
    value: "internal-ops",
    label: "Internal Operations",
  },
];

const stateOptions = serviceStates;

class ServiceInfo extends React.Component {
  render() {
    const { isEditing, service } = this.props;
    return (
      <Panel header="Service Overview">
        <ListGroup fill>
          <ListGroupItem>
            <Row>
              <Col md={6}>
                <Field
                  name="name"
                  component={FormText}
                  type="text"
                  title="Name"
                  isEditing={isEditing}
                />
                <UserSelect
                  reduxPath="primary_owner_id"
                  isEditing={isEditing}
                  user={service && service.primary_owner}
                  userID={service && service.primary_owner_id}
                  title="Primary Owner"
                />
                <br />
                {!isEditing && (
                  <ServiceDetailHierarchy
                    user_ids={[service.primary_owner.uid]}
                  />
                )}
              </Col>
              <Col md={6}>
                <Field
                  name="state"
                  component={FormSelectType}
                  title="Service State"
                  isEditing={isEditing}
                  options={stateOptions}
                />
                <Field
                  name="type"
                  component={FormSelectType}
                  title="Service Type"
                  isEditing={isEditing}
                  options={serviceOptions}
                />
                {((service && service.slack_channel) || isEditing) && (
                  <Field
                    name="slack_channel_id"
                    component={FormSelect}
                    title="Service Slack"
                    dataType="slack.channels"
                    labelKey="name"
                    valueKey="id"
                    query={slackChannelsQuery}
                    id="serviceSlack"
                    isEditing={isEditing}
                    linkProducer={() =>
                      service && toSlack(service.slack_channel_id)
                    }
                    currentValue={_.get(service, "slack_channel.name", "")}
                    additionalOptions={[{ label: "", value: "" }]}
                  />
                )}
                {((service && service.pagerduty) || isEditing) && (
                  <Field
                    name="pagerduty"
                    component={FormText}
                    type="text"
                    title="Service Pagerduty"
                    isEditing={isEditing}
                    linkProducer={() => service && service.pagerduty}
                  />
                )}
                {!isEditing &&
                  (!service.slack_channel || !service.pagerduty) && (
                    <div>
                      {!service.slack_channel && (
                        <h5>
                          <label>Service Slack: </label>&nbsp;**
                          <br />
                        </h5>
                      )}
                      {!service.pagerduty && (
                        <h5>
                          <label>Service Pagerduty: </label>&nbsp;**
                          <br />
                        </h5>
                      )}
                      (<i>** not configured, edit service to update</i>)
                    </div>
                  )}
              </Col>
            </Row>
          </ListGroupItem>
          <ListGroupItem>
            <FormAttributeSelectField
              name="attributes"
              id="serviceAttributesEdit"
              dataType="attribute"
              isEditing={isEditing}
              data={service ? service.attributes : []}
            />
          </ListGroupItem>
          <ListGroupItem>
            <Row>
              <Col md={12}>
                <Field
                  name="description"
                  component={FormTextArea}
                  type="text"
                  title="Description"
                  isEditing={isEditing}
                  maxLength={2000}
                />
              </Col>
            </Row>
          </ListGroupItem>
        </ListGroup>
      </Panel>
    );
  }
}

ServiceInfo.propTypes = {
  isEditing: PropTypes.bool,
  service: PropTypes.object,
};

export default ServiceInfo;
