import React from "react";
import PropTypes from "prop-types";
import ErrorMessage from "components/ErrorMessage";
import MessagingModal from "components/MessagingModal";
import {LoadingSpinner} from "components/LoadingSpinner";
import ServiceInfo from "./ServiceInfo";
import ServiceAudits from "./ServiceAudits";
import ServiceEndpointsWrapper from "./ServiceEndpoints";
import ServiceDependencies from "./ServiceDependencies";
import Actions from "components/Actions";
import { reduxForm, formValueSelector } from "redux-form";
import LogRecordList from "../../../common/LogRecord/components/LogRecordList";
import {ServiceNames, trimServiceName} from "./ServiceNames";
import {READ_ONLY_MODE} from "../../../constants";
import UserDataPrivacy from "./UserDataPrivacy";
import { connect } from "react-redux";


var currentServiceID = "0";
// store a list of existing services (to display its name and link to it when match is found)
export var services = [];
// store a list of trimmed service-names, to perform lookup when user inputs service-name in service-creation form
export var trimmedServiceNames = [];

export class ServiceView extends React.Component {
  render() {
    // handleSubmit provided by ReduxForm
    const {
      serviceID,
      service,
      upstreamSlacks,
      downstreamSlacks,
      isEditing,
      isFetching,
      toggleEditing,
      handleSubmit,
      valid,
      toggleServiceAuditModal,
      serviceAuditModal,
      createServiceAudit,
      reset,
      returnToIndex,
      createService,
      updateService,
      deleteService,
      storesUserDataSelected,
      accessExceptionRequired,
      deletionExceptionRequired,
      setFieldValue,
    } = this.props;

    let content;

    function updateFunc(serviceData) {
      if (serviceData.id != undefined) {
        updateService(serviceData);
      } else {
        createService(serviceData);
      }
    }

    if (isFetching) {
      content = <LoadingSpinner/>;
    } else if (!isEditing && service === undefined) {
      content = <ErrorMessage/>;
    } else {
      let logs = service ? service.logs : [];
      currentServiceID = service ? service.id : "0";
      if (isEditing && READ_ONLY_MODE) {
        return null;
      }
      content = (
        <div>
          {isEditing && <ServiceNames/>}
          <ServiceInfo service={service} isEditing={isEditing} />
          <UserDataPrivacy
            setFieldValue={setFieldValue}
            isEditing={isEditing}
            storesUserDataSelected={storesUserDataSelected}
            accessExceptionRequired={accessExceptionRequired}
            deletionExceptionRequired={deletionExceptionRequired}
          />
          <ServiceEndpointsWrapper isEditing={isEditing} />
          <ServiceDependencies
            serviceID={serviceID}
            service={service}
            isEditing={isEditing}
            upstreamSlacks={upstreamSlacks}
            downstreamSlacks={downstreamSlacks}
          />
          {!isEditing && (
            <ServiceAudits
              service={service}
              toggleModal={toggleServiceAuditModal}
              modalState={serviceAuditModal}
              isEditing={isEditing}
              createServiceAudit={createServiceAudit}
            />
          )}
          {!isEditing && <LogRecordList logs={logs}/>}
          {!READ_ONLY_MODE && (<Actions
              update={updateFunc}
              handleSubmit={handleSubmit}
              toggleEditing={toggleEditing}
              isEditing={isEditing}
              valid={valid}
              delete={() => deleteService(serviceID)}
              reset={() => {
                console.log("Resetting!", reset);
                if (serviceID === "new") {
                  returnToIndex();
                } else {
                  reset();
                }
              }}
            />
          )}
          <MessagingModal service={service}/>
        </div>
      );
    }
    return content;
  }
}

const emptyOrUndefined = value =>
  value === undefined ||
  (value !== undefined && value.toString().trim() === "");

// A Service validation function
// Docs: http://redux-form.com/6.0.2/examples/syncValidation/
const formValidation = values => {
  const errors = {};
  if (values.attributes) {
    var containsTier = false;
    values.attributes.forEach(attr => {
      if (attr.name == "tier" && !attr.deleted) {
        containsTier = true;
        if (!(attr.value == "1" || attr.value == "2")) {
          errors.attributes = "tier must be 1 or 2";
        }
      }
    });
    if (!containsTier) {
      errors.attributes = "Services must have tier set";
    }
  }
  if (values.type == "") {
    errors.type = "A Service Type must be selected.";
  }
  if (values.name == "") {
    errors.name = "Services must be named.";
  }

  // User Data Privacy
  if (emptyOrUndefined(values.stores_user_data)) {
    errors.stores_user_data =
      "Indicate whether your service stores personal user data.";
  }

  if (emptyOrUndefined(values.access_requests)) {
    errors.access_requests =
      "Indicate whether your service must respond to user data access request.";
  }

  if (emptyOrUndefined(values.access_exception)) {
    errors.access_exception = "Describe your Legal-Approved exception.";
  }

  if (emptyOrUndefined(values.access_exception_validated_by)) {
    errors.access_exception_validated_by =
      "Indicate who approved the exception.";
  }

  if (emptyOrUndefined(values.deletion_requests)) {
    errors.deletion_requests =
      "Indicate whether your service must respond to user data deletion request.";
  }

  if (emptyOrUndefined(values.deletion_exception)) {
    errors.deletion_exception = "Describe your Legal-Approved exception.";
  }

  if (emptyOrUndefined(values.deletion_exception_validated_by)) {
    errors.deletion_exception_validated_by =
      "Indicate who approved the exception.";
  }

  if (
    !values.user_data_types ||
    (values.user_data_types && !values.user_data_types.length)
  ) {
    errors.user_data_types = "Indicate the type of data your service stores.";
  }

  if (!values.primary_owner_id || values.primary_owner_id == "0") {
    errors.primary_owner_id = "Services must have a Primary Owner.";
  }
  // Check the service name being entered for a match with existing service after trim
  if (trimmedServiceNames && values.name && trimmedServiceNames.includes(trimServiceName(values.name))) {
    var matchingServiceID = "0";
    var serviceName = "unknown";
    for (let service of services) {
      serviceName = service.name;
      if (trimServiceName(service.name) === trimServiceName(values.name)) {
        matchingServiceID = service.id.toString();
        break;
      }
    }
    if (matchingServiceID !== currentServiceID) {
      // generate link to the existing service that was found matching
      let matchingServiceURL = "";
      if (document.location.href.indexOf("new?edit=1") !== -1) {
        // user is in create service flow
        matchingServiceURL = document.location.href.replace("new?edit=1", matchingServiceID);
      } else {
        // user is in edit service flow
        matchingServiceURL = document.location.href.replace(currentServiceID + "?edit=1", matchingServiceID);
      }
      errors.name = <div>Service already exists with that name: {serviceName}, see <a target={"_blank"}
                                                                                      href={matchingServiceURL}>here</a>
      </div>;
    }
  }
  if (
    values.pagerduty &&
    !values.pagerduty.match(
      /^https:\/\/twitchoncall.pagerduty.com\/services\/P[A-Z0-9]{6}$/
    ) &&
    !values.pagerduty.match(/^mailto:page-[a-z0-9_-]+@amazon.com$/)
  ) {
    errors.pagerduty =
      "Pagerduty service URLs should be in the form of https://twitchoncall.pagerduty.com/services/ABCDEFG. Amazon oncall emails in the form of mailto:page-...@amazon.com are also accepted.";
  }

  // See https://redux-form.com/8.2.2/examples/fieldarrays/
  if (values.endpoints) {

    // Determine all endpoint-level errors
    var errorsArray = values.endpoints.map(function (endpoint) {
      const endpointErrors = {};
      if (!endpoint.stage || endpoint.stage.length === 0) {
        endpointErrors.stage = 'Required';
      }
      if (!endpoint.region || endpoint.region.length === 0) {
        endpointErrors.region = 'Required';
      }
      if (!endpoint.hostname || endpoint.hostname.length === 0) {
        endpointErrors.hostname = 'Required';
      }
      if (!endpoint.vpc_endpoint_service_id || endpoint.vpc_endpoint_service_id.length === 0) {
        endpointErrors.vpc_endpoint_service_id = 'Required';
      }
      if (!endpoint.port || endpoint.port < 1) {
        endpointErrors.port = 'Nonnegative';
      }
      return endpointErrors;
    });

    if (errorsArray.length) {
      errors.endpoints = errorsArray;
    }
  }
  return errors;
};

ServiceView.propTypes = {
  service: PropTypes.object,
  createService: PropTypes.func.isRequired,
  updateService: PropTypes.func.isRequired,
  deleteService: PropTypes.func.isRequired,
  isEditing: PropTypes.bool.isRequired,
  reset: PropTypes.func.isRequired,
};

// A wrapper to support the edit mode for the service
// view page. This adds logic to wire together <Field/>
// elements with redux state, and a callback when a submit
// action is triggered.  Won't make any difference when
// the view is not in an "edit" mode
var ServiceViewWrapper = reduxForm({
  form: "serviceEdit",
  validate: formValidation,
})(ServiceView);

// Selector to use the values from the form
const selector = formValueSelector("serviceEdit");
ServiceViewWrapper = connect(state => {
  const storesUserDataSelected = selector(state, "stores_user_data");
  const accessExceptionRequired = selector(state, "access_requests");
  const deletionExceptionRequired = selector(state, "deletion_requests");

  return {
    storesUserDataSelected,
    accessExceptionRequired:
      storesUserDataSelected === true && accessExceptionRequired === false,
    deletionExceptionRequired:
      storesUserDataSelected === true && deletionExceptionRequired === false,
  };
})(ServiceViewWrapper);

export default ServiceViewWrapper;
