// @format
import React from "react";
import { connect } from "react-redux";
import _ from "lodash";
import { addUrlProps, UrlQueryParamTypes } from "react-url-query";
import {
  serviceAuditModalToggle,
  serviceViewReturnToIndex,
  serviceMetricModalToggle,
} from "../modules/service_view";
import {
  fetchServiceIfNeeded,
  createService,
  updateService,
  deleteService,
  fetchTeamIfNeeded,
  createServiceAudit,
} from "common/goracle/actions";
import { initialize, change } from "redux-form";
import { metricQuerySelectedAction } from "store/uiReducer";
import { createSelector } from "reselect";
import { setMessageReplyTo } from "store/messaging";

/*  This is a container component. Notice it does not contain any JSX,
    nor does it import React. This component is **only** responsible for
    wiring in the actions and state necessary to render a presentational
    component - in this case, the counter:   */

import ServiceView from "../components/ServiceView";
import LoginComponent from "../../Login/LoginComponent";

/*  Object of action creators (can also be function that returns object).
    Keys will be passed as props to presentational components. Here we are
    implementing our wrapper around increment; the component doesn't care   */

const urlPropsQueryConfig = {
  edit: { type: UrlQueryParamTypes.boolean },
};

const defaultService = {
  name: "",
  state: "Active",
  team_id: "0",
  type: "",
  description: "",
  pagerduty: "",
  slack: "",
  component_ids: [],
  components: [],
  attributes: [{ name: "tier", value: "2" }],
  service_upstreams: [],
  service_downstreams: [],
  endpoints: [],
  access_requests: true,
  deletion_requests: true,
};

export class ServiceViewContainer extends React.Component {
  constructor(props) {
    super(props);
    this.state = {
      loading: false
    }
  }
  componentWillMount() {
    const id = this.props.match.params.id;
    const { fetchServiceIfNeeded, formInitialize, servicesByID } = this.props;
    const serviceData = servicesByID[id];
    if (id != "new") {
      fetchServiceIfNeeded(id).catch(error => {
        console.log(`Unable to fetch service ${id}!`, error);
        console.log("redirecting to Service Index page...");
        this.props.history.push('/');
      });
      formInitialize(serviceData ? serviceData.service : undefined);
    } else {
      this.props.onChangeEdit(true);
      this.props.formInitialize();
    }
  }

  componentWillReceiveProps(nextProps) {
    // All of these variables and code exist in order to determine whether
    // the form fields need to be updated to new values based on prop changes.
    // This is mostly to make the logic below clearer, it's not strictly necessary
    // since most of these variables are used only once.
    const oldID = this.props.match.params.id;
    const newID = nextProps.match.params.id;
    const oldServicesByID = this.props.servicesByID;
    const newServicesByID = nextProps.servicesByID;
    const oldEdit = this.props.edit;
    const newEdit = this.props.edit;
    const formInitialize = this.props.formInitialize;
    const oldServiceData = oldServicesByID ? oldServicesByID[oldID] : undefined;
    const newServiceData = newServicesByID ? newServicesByID[newID] : undefined;
    const oldReceived = oldServiceData ? oldServiceData.receivedAt : undefined;
    const newReceived = newServiceData ? newServiceData.receivedAt : undefined;

    let fillForm = false;
    if (oldID != newID) {
      // Navigating to a new service
      if (newID != "new") {
        this.props.fetchServiceIfNeeded(newID).catch(error => {
          console.log(`Unable to fetch service ${newID}!`, error);
          console.log("redirecting to Service Index page...");
          this.props.history.push('/');
        });
      }
      fillForm = true;
    } else if (oldReceived !== newReceived) {
      // Data has changed (refreshed from network)
      fillForm = true;
    } else if (newEdit == false && oldEdit == true) {
      // Editing was cancelled
      fillForm = true;
    }

    if (fillForm) {
      // Update form with data from goracle (via props)
      formInitialize(newServiceData ? newServiceData.service : undefined);
    }
  }

  componentDidUpdate() {
    if (this.props.match.params.id != "new") {
      this.props
        .fetchServiceIfNeeded(this.props.match.params.id)
        .catch(error => {
          console.log(
            `Unable to fetch service ${this.props.match.params.id}!`,
            error
          );
          console.log("redirecting to Service Index page...");
          this.props.history.push('/');
        });
      if (
        this.props.servicesByID[this.props.match.params.id] &&
        this.props.servicesByID[this.props.match.params.id].service
      ) {
        let service = this.props.servicesByID[this.props.match.params.id]
          .service;

        if (service.team) {
          this.props.fetchTeamIfNeeded(service.team.id);
        }
        if (service.service_upstreams) {
          service.service_upstreams.forEach(dep => {
            if (dep.root_service) {
              this.props.fetchServiceIfNeeded(dep.root_service.id);
              let state = this.props.servicesByID[dep.root_service.id];
              if (state && state.service) {
                let upService = state.service;
                if (upService.team) {
                  this.props.fetchTeamIfNeeded(upService.team.id);
                }
              }
            }
          });
        }
        if (service.service_downstreams) {
          service.service_downstreams.forEach(dep => {
            if (dep.downstream_service) {
              this.props
                .fetchServiceIfNeeded(dep.downstream_service.id)
                .catch(error => {
                  console.log(`Unable to fetch service ${newID}!`);
                  // FIXME: Do something appropriate in the UI
                });
              let state = this.props.servicesByID[dep.downstream_service.id];
              if (state && state.service) {
                let downService = state.service;
                if (downService.team) {
                  this.props.fetchTeamIfNeeded(downService.team.id);
                }
              }
            }
          });
        }
      }
    }
  }

  // As little presentational logic as possible goes here.
  // Use some basic conditionals to decide which visual components should be rendered
  // based on the current state
  render() {
    const {
      servicesByID,
      teamsByID,
      upstreamSlacks,
      downstreamSlacks,
      serviceMetricModal,
      serviceAuditModal,
      edit,
      onChangeEdit,
      history,
      createService,
      updateService,
      deleteService,
      createServiceAudit,
      toggleServiceAuditModal,
      returnToIndex,
      toggleServiceMetricModal,
      setMessageReplyTo,
      setFieldValue,
    } = this.props;
    let serviceID = this.props.match.params.id;
    let serviceData = servicesByID ? servicesByID[serviceID] : undefined;
    let teamData;
    if (
      serviceData &&
      serviceData.service &&
      serviceData.service.team &&
      serviceData.service.team.id
    ) {
      teamData = teamsByID ? teamsByID[serviceData.service.team.id] : undefined;
    }

    let {
      service = undefined,
      detailed = false,
      isFetching = false,
      isError = false,
      status = false,
    } =
      serviceData || {};
    let {loading} = this.state;

    function toggleEdit(editing) {
      // Set edit to undefined instead of false if editing is false
      editing ? onChangeEdit(true) : onChangeEdit(undefined);
    }

    function createFunc(serviceData) {
      comp.setState({loading: true})
      createService(serviceData)
        .then(service => {
          comp.setState({loading: false})
          // Navigate to new location
          onChangeEdit(undefined);
          history.push(`/services/${service.id}`);
        })
        .catch(error => {
          comp.setState({loading: false})
          console.log("error creating service", error);
        });
    }

    let comp = this;
    function updateFunc(serviceData) {
      comp.setState({loading: true})
      updateService(serviceData)
        .then(service => {
          comp.setState({loading: false})
          onChangeEdit(undefined);
        })
        .catch(error => {
          comp.setState({loading: false})
          console.log("error updating service", error);
        });
    }

    function deleteFunc(serviceID) {
      comp.setState({loading: true})
      deleteService(serviceID)
        .then(service => {
          comp.setState({loading: false})
          onChangeEdit(undefined);
          history.push("/services/");
        })
        .catch(error => {
          comp.setState({loading: false})
          console.log("error deleting service", error);
        });
    }

    return (
      <div>
        <LoginComponent/>
        <ServiceView
          service={service}
          detailed={detailed}
          isError={isError}
          status={status}
          team={teamData ? teamData.team : {}}
          upstreamSlacks={upstreamSlacks}
          downstreamSlacks={downstreamSlacks}
          serviceID={serviceID}
          isEditing={!!edit}
          isFetching={_.get(serviceData, "isFetching", false) || loading}
          serviceMetricModal={serviceMetricModal}
          serviceAuditModal={serviceAuditModal}
          toggleEditing={toggleEdit}
          createService={createFunc}
          updateService={updateFunc}
          createServiceAudit={createServiceAudit}
          toggleServiceAuditModal={toggleServiceAuditModal}
          deleteService={deleteFunc}
          returnToIndex={returnToIndex}
          toggleServiceMetricModal={toggleServiceMetricModal}
          setMessageReplyTo={setMessageReplyTo}
          setFieldValue={setFieldValue}
        />
      </div>
    );
  }
}

const getCurrentServiceId = (state, props) => props.match.params.id;
const getServices = (state, props) => state.goracle.servicesByID;

var upstreamServicesContactsLoaded = false;

// function to decide whether to display slack link for upstream services
// only set to true once all upstreams are loaded
export function displayUpstreamSlackLink() {
  return upstreamServicesContactsLoaded
}

const getUpstreamServicesContacts = createSelector(
  getCurrentServiceId,
  getServices,
  (id, services) => {
    let upstreams = _.get(services[id], "service.service_upstreams");
    if (upstreams) {
      let slacks = upstreams.map(dep => {
        if (
          dep.root_service &&
          _.has(services[dep.root_service.id], "service.slack_channel.name")
        ) {
          return services[dep.root_service.id].service.slack_channel.name;
        }
      });
      slacks = _.uniq(_.compact(slacks));
      // set upstreamServicesContactsLoaded to true if there's at least one downstream service
      // services.length is 2 if there are no upstreams
      if (Object.keys(services).length > 2) {
        upstreamServicesContactsLoaded = true;
      }
      // set upstreamServicesContactsLoaded to false if there's any upstream service still fetching
      // to indicate that loading has not completed
      Object.keys(services).map(function(keyName) {
        if (services[keyName].isFetching === true) {
          upstreamServicesContactsLoaded = false;
        }
      });
      return { slacks };
    }
  }
);

var downstreamServicesContactsLoaded = false;

// function to decide whether to display slack link for downstream services
// only set to true once all downstreams are loaded
export function displayDownstreamSlackLink() {
  return downstreamServicesContactsLoaded
}

const getDownstreamServicesContacts = createSelector(
  getCurrentServiceId,
  getServices,
  (id, services) => {
    let downstreams = _.get(services[id], "service.service_downstreams");
    if (downstreams) {
      let slacks = downstreams.map(dep => {
        if (
          dep.downstream_service &&
          _.has(services[dep.downstream_service.id], "service.slack_channel.name")
        ) {
          return services[dep.downstream_service.id].service.slack_channel.name;
        }
      });
      slacks = _.uniq(_.compact(slacks));
      // set downstreamServicesContactsLoaded to true if there's at least one downstream service
      // services.length is 2 if there are no downstreams
      if (Object.keys(services).length > 2) {
        downstreamServicesContactsLoaded = true;
      }
      // set downstreamServicesContactsLoaded to false if there's any downstream service still fetching
      // to indicate that loading has not completed
      Object.keys(services).map(function(keyName) {
        if (services[keyName].isFetching === true) {
          downstreamServicesContactsLoaded = false;
        }
      });
      return { slacks };
    }
  }
);

var mapDispatchToProps = function(dispatch) {
  return {
    fetchServiceIfNeeded: id => {
      return dispatch(fetchServiceIfNeeded(id));
    },
    fetchTeamIfNeeded: id => {
      return dispatch(fetchTeamIfNeeded(id));
    },
    formInitialize: service =>
      dispatch(initialize("serviceEdit", _.merge({}, defaultService, service))),
    createService: serviceData => {
      return dispatch(createService(serviceData));
    },
    updateService: serviceData => {
      return dispatch(updateService(serviceData));
    },
    deleteService: serviceID => {
      return dispatch(deleteService(serviceID));
    },
    createServiceAudit: (serviceID, auditType, action, auditValue) => {
      return dispatch(createServiceAudit(serviceID, auditType, action, auditValue));
    },
    toggleServiceAuditModal: (enabled, auditType) =>
      dispatch(serviceAuditModalToggle(enabled, auditType)),
    returnToIndex: () => {
      dispatch(serviceViewReturnToIndex());
    },
    toggleServiceMetricModal: (enabled, metric) =>
      dispatch(dispatch => {
        if (metric.latency_query) {
          dispatch(metricQuerySelectedAction(metric.latency_query));
        }
        dispatch(serviceMetricModalToggle(enabled, metric));
      }),
    setMessageReplyTo: email => dispatch(setMessageReplyTo(email)),
    setFieldValue: (field, value) => {
      dispatch(change("serviceEdit", field, value));
    },
  };
};

var mapStateToProps = function(state, props) {
  const { servicesByID, teamsByID } = state.goracle;
  const upstreamSlacks = getUpstreamServicesContacts(state, props);
  const downstreamSlacks = getDownstreamServicesContacts(state, props);
  const {
    hasErrored,
    serviceAuditModal,
    serviceMetricModal,
  } = state.service_view;
  return {
    servicesByID: servicesByID,
    teamsByID: teamsByID,
    hasErrored: hasErrored,
    upstreamSlacks: upstreamSlacks,
    downstreamSlacks: downstreamSlacks,
    serviceAuditModal: serviceAuditModal,
    serviceMetricModal: serviceMetricModal,
  };
};

export default addUrlProps({ urlPropsQueryConfig })(
  connect(mapStateToProps, mapDispatchToProps)(ServiceViewContainer)
);
