import { combineReducers } from "redux";
import { GraphqlQuery, GraphqlMutation } from "lib/graphql";
import { initialize as formInitialize } from "redux-form";
import { push } from "react-router-redux";
import { toService } from "lib/formatters";
import gql from "graphql-tag";

// ------------------------------------
// Constants
// ------------------------------------
export const SERVICES_FETCH_DATA_SUCCESS = "SERVICES_FETCH_DATA_SUCCESS";
export const SERVICES_HAS_ERRORED = "SERVICES_HAS_ERRORED";
export const SERVICES_IS_LOADING = "SERVICES_IS_LOADING";
export const SERVICES_IS_CREATING = "SERVICES_IS_CREATING";

// ------------------------------------
// GraphQL Queries
// ------------------------------------
const serviceListQuery = gql`
  query Services {
    services {
      id
      name
      team {
        id
        name
        org {
          id
          name
        }
      }
      attributes {
        name
        value
      }
      latest_service_audits {
        id
        audit_type
        auditor
        action
        audit_time
      }
    }
  }
`;

// ------------------------------------
// Actions
// ------------------------------------
export function servicesHasErroredAction(bool) {
  return {
    type: SERVICES_HAS_ERRORED,
    hasErrored: bool,
  };
}
export function servicesIsLoadingAction(bool) {
  return {
    type: SERVICES_IS_LOADING,
    isLoading: bool,
  };
}

export function serviceCreate() {
  return dispatch => {
    dispatch(push(toService("new")));
  };
}

// GraphQL response data returned from an API call
export function servicesFetchDataSuccess(servicesData) {
  return {
    type: SERVICES_FETCH_DATA_SUCCESS,
    services: servicesData.services,
  };
}

/*  This is a thunk, meaning it is a function that immediately
    returns a function for lazy evaluation. It is incredibly useful for
    creating async actions, especially when combined with redux-thunk! */

export function servicesFetchData() {
  return dispatch => {
    GraphqlQuery(
      serviceListQuery,
      {}, // no vars
      servicesIsLoadingAction,
      servicesHasErroredAction,
      servicesFetchDataSuccess,
      dispatch
    );
  };
}

export const actions = {
  servicesFetchData,
  servicesFetchDataSuccess,
  servicesIsLoading,
  servicesHasErrored,
};

// ------------------------------------
// Reducers
// ------------------------------------

export function servicesHasErrored(state = false, action) {
  switch (action.type) {
    case SERVICES_HAS_ERRORED:
      return action.hasErrored;
    default:
      return state;
  }
}
export function servicesIsLoading(state = false, action) {
  switch (action.type) {
    case SERVICES_IS_LOADING:
      return action.isLoading;
    default:
      return state;
  }
}
export function servicesIsCreating(state = false, action) {
  switch (action.type) {
    case SERVICES_IS_CREATING:
      return action.isCreating;
    default:
      return state;
  }
}

const initialServicesData = {
  services: [],
};
export function servicesData(state = initialServicesData, action) {
  switch (action.type) {
    case SERVICES_FETCH_DATA_SUCCESS:
      // return the new list of services without changing filters
      return {
        services: action.services,
      };
    default:
      return state;
  }
}

export default combineReducers({
  servicesData,
  servicesHasErrored,
  servicesIsLoading,
  servicesIsCreating,
});
