import * as libGraphql from "lib/graphql";
import { makeExecutableSchema, addMockFunctionsToSchema } from "graphql-tools";
import { graphql } from "graphql";
import gql from "graphql-tag";

import { schemaString } from "lib/__mocks__/graphqlSchema";

jest.unmock("graphql");

const query = gql`
  query MockQuery($id: ID!) {
    account(id: $id) {
      id
      alias
    }
  }
`;

function mockFetchGraphqlOnce() {
  const schema = makeExecutableSchema({ typeDefs: schemaString });
  const mocks = {
    ID: () => "1",
  };
  addMockFunctionsToSchema({ schema, mocks });

  fetch.once("*", function(url, opts) {
    let body = JSON.parse(opts.body);
    let graphqlOpts = {
      schema: schema,
      source: body.query,
      operationName: body.operationName,
      variableValues: body.variables,
    };
    return graphql(graphqlOpts);
  });
}

function mockFetchGraphqlFailOnce() {
  fetch.once("*", {
    status: 502,
    body: {},
  });
}

describe("graqhql", () => {
  afterEach(() => {
    fetch.reset();
    fetch.restore();
  });

  test("graphqlQueryPromise should resolve a promise on success with data", () => {
    mockFetchGraphqlOnce();

    expect.assertions(1);
    let variables = {
      id: 1,
    };
    return expect(
      libGraphql.graphqlQueryPromise({
        query: query,
        variables: variables,
      })
    ).resolves.toMatchObject({
      data: {
        account: {
          id: "1",
        },
      },
    });
  });

  test("graphqlQueryPromise should throw an error on failure", () => {
    mockFetchGraphqlFailOnce();

    // Disable console.log and console.error to reduce spam on this test
    // console.log = jest.fn();
    // console.error = jest.fn();

    expect.assertions(1);
    let variables = {
      id: 1,
    };

    console.log("Testing GraphQl Error case, error print of 502 is expected")
    return expect(
      libGraphql.graphqlQueryPromise({
        query: query,
        variables: variables,
      })
    ).rejects.toEqual(expect.any(Error));
  });
});
