package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"strings"
	"time"
)

type timestamp struct {
	time.Time
}

func (u *timestamp) UnmarshalJSON(b []byte) error {
	var val int64

	if err := json.Unmarshal(b, &val); err != nil {
		return err
	}

	u.Time = time.Unix(val, 0)

	return nil
}

// VPNRecord represents a connection line from an OpenVPN AS log file.
type VPNRecord struct {
	Username   string    `json:"username"`
	BytesTotal int       `json:"bytes_total"`
	Node       string    `json:"node"`
	Service    string    `json:"service"`
	Proto      string    `json:"proto"`
	BytesIn    int       `json:"bytes_in"`
	StartTime  timestamp `json:"start_time"`
	RealIP     string    `json:"real_ip"`
	SessionID  string    `json:"session_id"`
	Active     *int      `json:"active,omitempty"`
	VpnIP      string    `json:"vpn_ip"`
	Timestamp  int       `json:"timestamp"`
	Auth       *int      `json:"auth,omitempty"`
	BytesOut   int       `json:"bytes_out"`
	Duration   *int      `json:"duration"`
	CommonName string    `json:"common_name,omitempty"`
	Port       string    `json:"port"`
	Error      string    `json:"error,omitempty"`
}

func buildUserByDayReport(records []*VPNRecord) (time.Time, time.Time, map[string]map[string][]*VPNRecord) {
	start := time.Now()
	end := time.Time{}

	byuserday := make(map[string]map[string][]*VPNRecord)
	for _, user := range records {
		username := strings.TrimSuffix(strings.TrimSuffix(strings.TrimSuffix(
			strings.TrimSpace(user.Username), "@justin.tv"), "@twitch.tv"), "@AMZN")
		if byuserday[username] == nil {
			byuserday[username] = make(map[string][]*VPNRecord)
		}

		if end.Before(user.StartTime.Time) {
			end = user.StartTime.Time
		} else if start.After(user.StartTime.Time) {
			start = user.StartTime.Time
		}

		date := user.StartTime.Format("2006-01-02")
		byuserday[username][date] = append(byuserday[username][date], user)
	}

	return start, end, byuserday
}

func parseFiles() ([]*VPNRecord, error) {
	users := []*VPNRecord{}

	for _, file := range inputFiles {
		f, err := os.Open(file)
		if err != nil {
			log.Println("Skipping missing file:", file)
			continue
		}
		defer f.Close() // this is in a loop. :(

		data, err := ioutil.ReadAll(f)
		if err != nil {
			log.Println("Skipping unreadable file:", file)
			continue
		}

		for i, line := range bytes.Split(bytes.TrimSpace(data), []byte("\n")) {
			record := &VPNRecord{}

			if err := json.Unmarshal(line, record); err != nil {
				return nil, fmt.Errorf("line %d in file %s: %w (%s)", i, file, err, string(line))
			}

			users = append(users, record)
		}
	}

	return users, nil
}
