package main

import (
	"fmt"
	"sort"
	"strings"
	"time"
)

const oneDay = 24 * time.Hour

// outputUserByDayCSV outputs one huge line per user.
func outputUserByDayCSV(start, end time.Time, report map[string]map[string][]*VPNRecord) []string {
	lines := []string{"username,"}
	header := "username,"

	for i := start; i.Before(end); i = i.Add(oneDay) {
		if header[len(header)-1] != ',' {
			header += ","
		}

		date := i.Format("2006-01-02")
		header += date + ",dur,min,max,avg"
	}

	for username, data := range report {
		if username == "" {
			continue
		}

		line := username + ","

		for i := start; i.Before(end); i = i.Add(oneDay) {
			var dur, min, max time.Duration

			if line[len(line)-1] != ',' {
				line += ","
			}

			date := i.Format("2006-01-02")
			if data[date] == nil {
				line += "0,0,0,0,0"
				continue
			}

			for _, con := range data[date] {
				if con.Duration != nil {
					this := time.Second * time.Duration(*con.Duration)
					if (this < min || min == 0) && this != 0 {
						min = this
					}
					if this > max {
						max = this
					}
					dur += this
				}
			}

			avg := dur / time.Duration(len(data[date]))
			line += fmt.Sprintf("%d,%.0f,%.0f,%.0f,%.0f", len(data[date]), dur.Minutes(), min.Minutes(), max.Minutes(), avg.Minutes())
		}

		lines = append(lines, line)
	}

	sort.Strings(lines)

	return append([]string{header}, lines...)
}

func outputErrorsByDayCSV(start, end time.Time, report map[string]map[string][]*VPNRecord) []string {
	lines := []string{}
	header := "username"

	for i := start; i.Before(end); i = i.Add(oneDay) {
		header += "," + i.Format("2006-01-02")
	}

	for username, data := range report {
		line := strings.TrimSpace(username)
		if line != "" {
			for i := start; i.Before(end); i = i.Add(oneDay) {
				line += fmt.Sprintf(",%d", len(data[i.Format("2006-01-02")]))
			}

			lines = append(lines, line)
		}
	}
	sort.Strings(lines)

	return append([]string{header}, lines...)
}
