package main

import (
	"flag"
	"fmt"
	"log"
	"path"
	"strings"
	"time"

	"git.xarth.tv/awsi/eniflowner/pkg/eni"
	"git.xarth.tv/awsi/eniflowner/pkg/funcs"
	"git.xarth.tv/awsi/eniflowner/pkg/report"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/cloudwatchlogs"
)

//nolint:gochecknoglobals
var (
	// These ports are ignored. puppet, statsd, graphite, ldap, nagios.
	filterPorts = funcs.Filter{8140, 2003, 8125, 636, 19237, 19238}
	// All of the log streams for our 8 VPN servers.
	streams = []*eni.Stream{{
		Region: "us-east-1",
		Group:  "vpn-eni-flow-logs",
		Name:   "eni-0b9bb8e050f30b4f8-accept",
		IP:     "10.160.2.75",
	}, {
		Region: "us-east-1",
		Group:  "vpn-eni-flow-logs",
		Name:   "eni-0aba6df4642d62519-accept",
		IP:     "10.160.0.254",
	}, {
		Region: "us-west-2",
		Group:  "vpn-eni-flow-logs",
		Name:   "eni-0fc9b2205c0759559-accept",
		IP:     "10.197.192.68",
	}, {
		Region: "us-west-2",
		Group:  "vpn-eni-flow-logs",
		Name:   "eni-042f4a6f74f2a5426-accept",
		IP:     "10.197.202.23",
	}, {
		Region: "eu-west-1",
		Group:  "vpn-eni-flow-logs",
		Name:   "eni-03acafaa4db94333d-accept",
		IP:     "10.176.211.87",
	}, {
		Region: "eu-west-1",
		Group:  "vpn-eni-flow-logs",
		Name:   "eni-0055d13cdbab78efe-accept",
		IP:     "10.176.211.61",
	}, {
		Region: "ap-southeast-1",
		Group:  "vpn-eni-flow-logs",
		Name:   "eni-0c94deba18d2474a9-accept",
		IP:     "10.166.23.151",
	}, {
		Region: "ap-southeast-1",
		Group:  "vpn-eni-flow-logs",
		Name:   "eni-0ee1f64ceaa6fbf1c-accept",
		IP:     "10.166.23.62",
	}}
)

func main() {
	log.SetFlags(log.LstdFlags)
	ParseFlags().Work()
}

// Flags is the input data.
type Flags struct {
	HTML  string
	JSON  string
	CSV   string
	IPMap string
	Load  funcs.StringSlice
	Age   time.Duration
	Print bool
	Type  string
}

// ParseFlags turns the input data into a struct.
func ParseFlags() *Flags {
	f := &Flags{}

	flag.DurationVar(&f.Age, "age", time.Hour, "How far back in history to scan CWL")
	flag.StringVar(&f.HTML, "html", "", "Provide path to write HTML file(s).")
	flag.StringVar(&f.JSON, "json", "", "Provide path to write JSON file.")
	flag.StringVar(&f.CSV, "csv", "", "Provide path to write CSV file.")
	flag.Var(&f.Load, "load", "Paths to JSON report files to combine; separate with comma.")
	flag.StringVar(&f.IPMap, "ipmap", "docs/ipmap.yaml", "Provide path to IPMap YAML file.")
	flag.StringVar(&f.Type, "type", "normal", "Report type; options: normal, account")
	flag.BoolVar(&f.Print, "print", false, "Use this to print flows to stdout.")
	flag.Parse()

	return f
}

// Work does the thing.
func (f *Flags) Work() {
	start := time.Now()

	oldReport, err := report.Load(f.Load)
	if err != nil {
		log.Println("Loading Old Report:", err)
	}

	log.Printf("Loaded %d old flows from '%s'. Grabbing %d log streams events from past %v (now: %v)",
		len(oldReport.Flows), f.Load, len(streams), f.Age, start.Round(time.Second))

	all, err := f.getEvents(streams)
	if err != nil {
		log.Fatal(err)
	}

	ipmap, err := funcs.GetIPMap(f.IPMap)
	if err != nil {
		log.Fatalln("Error Getting IPMap:", err)
	}

	if f.Age != 0 {
		log.Printf("Found %d total events over past %v in %v from %d streams. Aggregating them. Known IPs: %d",
			len(all), f.Age, time.Since(start).Round(time.Second), len(streams), len(ipmap))
	}

	keep, skip, filter, agg, who := funcs.AggregateEvents(all, filterPorts)

	log.Printf("Processing elapsed %v. Skipped %d unusable or external events and filtered %d."+
		" The remaining events %d aggregated to %d destinations.",
		time.Since(start).Round(time.Second), skip, filter, keep, len(agg))

	flows, newips := funcs.AggregateFlows(agg, who, ipmap)

	switch strings.ToLower(f.Type) {
	case "normal", "":
		f.normalOutput(&report.Report{
			Start:        start,
			Age:          f.Age,
			Streams:      len(streams),
			OldFlows:     len(oldReport.Flows),
			Events:       len(all),
			IPmap:        len(ipmap),
			Keep:         keep,
			Skip:         skip,
			Filtered:     filter,
			Destinations: len(agg),
			NewIPS:       len(newips),
			Files:        oldReport.Files,
			Flows:        funcs.CombineFlows(flows, oldReport.Flows),
		})
	case "account", "accounts":
		f.accountOutput(&report.Report{
			Start:        start,
			Age:          f.Age,
			Streams:      len(streams),
			OldFlows:     len(oldReport.Flows),
			Events:       len(all),
			IPmap:        len(ipmap),
			Keep:         keep,
			Skip:         skip,
			Filtered:     filter,
			Destinations: len(agg),
			NewIPS:       len(newips),
			Files:        oldReport.Files,
			Flows:        funcs.CombineFlows(flows, oldReport.Flows),
		})
	default:
		panic("invalid -type provided " + f.Type)
	}

	if err := funcs.UpdateIPMap(ipmap, newips, f.IPMap); err != nil {
		log.Println("Error Updating IPMap:", err)
	}
}

func (f *Flags) getEvents(streams []*eni.Stream) ([]*eni.Event, error) {
	all := []*eni.Event{}

	if f.Age == 0 {
		return all, nil
	}

	for _, stream := range streams {
		start := time.Now()

		cwl := cloudwatchlogs.New(session.Must(session.NewSessionWithOptions(session.Options{
			SharedConfigState: session.SharedConfigEnable,
			Config:            aws.Config{Region: &stream.Region},
		})))

		batches, events, err := eni.GetEvents(cwl, stream, f.Age)
		if err != nil {
			return nil, fmt.Errorf("getting events: %w", err)
		}

		log.Printf("Elapsed %v grabbing %d events (%d batches) from (%s) '%s:%s' ip %s",
			time.Since(start).Round(time.Second), len(events), batches, stream.Region, stream.Group, stream.Name, stream.IP)

		all = append(all, events...)
	}

	return all, nil
}

func (f *Flags) accountOutput(report *report.Report) {
	if f.CSV == "" && f.HTML == "" && f.JSON == "" {
		f.Print = true
	}

	accounts := report.GetAccounts()

	if f.Print {
		fmt.Printf("%-12s %-10s %-5s %-5s %-5s %s\n",
			"Isengard", "Owner", "IPs", "Ports", "Svcs", "Account")

		// This is not very usefule atm.
		for _, a := range accounts {
			fmt.Printf("%-12s %-10s %-5d %-5d %-5d %s\n",
				a.AccountID, a.Meta.PrimaryOwner, len(a.IPs), len(a.Ports), len(a.Svcs), a.Name)
		}
	}

	if f.HTML != "" {
		file := path.Join(f.HTML, "index")
		if err := report.WriteAccountHTMLIndex(file, accounts); err != nil {
			log.Println("Error writing HTML:", err)
		}

		for _, account := range accounts {
			file = path.Join(f.HTML, account.AccountID)
			if err := report.WriteAccountHTML(file, account); err != nil {
				log.Println("Error writing HTML:", err)
			}
		}
	}
}

func (f *Flags) normalOutput(report *report.Report) {
	if f.CSV == "" && f.HTML == "" && f.JSON == "" {
		f.Print = true
	}

	if f.Print {
		for _, f := range report.Flows {
			fmt.Printf("%-12s %-25s %-5d %-20s %s\n", f.AccountID, f.Name, f.Count, f.IPPort, f.Service)
		}
	}

	if err := report.WriteCSV(f.CSV); err != nil {
		log.Println("Error writing CSV:", err)
	}

	if err := report.WriteJSON(f.JSON); err != nil {
		log.Println("Error writing JSON:", err)
	}

	if err := report.WriteHTML(f.HTML); err != nil {
		log.Println("Error writing HTML:", err)
	}
}
