// Package eni provides procedures to download VPC Flow logs for specific ENIs,
// and turn them into Flows.
package eni

import (
	"fmt"
	"net"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/systems/find_ip_owner/pkg/cidrmap"
	"code.justin.tv/systems/find_ip_owner/pkg/isengard"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/cloudwatchlogs"
)

// Stream defines a cloudwatch logs stream we are getting eni flow log data from.
type Stream struct {
	Region string
	Group  string
	Name   string
	IP     string
}

// Flow defines a VPC flow log entry from cloudwatch.
// Format = "${version} ${account-id} ${interface-id} ${srcaddr} ${dstaddr} ${srcport} ${dstport} ${protocol} ${packets} ${bytes} ${start} ${end} ${action} ${log-status}"
// 2 007917851548  eni-0fc9b2205c0 10.195.64.170 10.197.192.68 443 56817 6 34 35008 1619600584 1619600603 ACCEPT OK
type Flow struct {
	Version   float64
	Account   string
	Interface string
	SrcAddr   string
	DstAddr   string
	SrcPort   int
	DstPort   int
	Protocol  string
	Packets   int64
	Bytes     int64
	Start     time.Time
	End       time.Time
	Action    string
	Status    string
	// The rest are not part of the normal Flow event.
	RemAddr string
	RemPort int
	Remote  isengard.AWSAccount
}

// Event is an event from cloudwatch logs. A log line, basically. We add a local IP to it.
type Event struct {
	LocalIP string
	*cloudwatchlogs.OutputLogEvent
}

func GetEvents(cwl *cloudwatchlogs.CloudWatchLogs, stream *Stream, dur time.Duration) (int, []*Event, error) {
	events := []*Event{}
	batches := 0

	err := cwl.GetLogEventsPages(&cloudwatchlogs.GetLogEventsInput{
		StartTime:     aws.Int64(time.Now().Add(-dur).UnixNano() / int64(time.Millisecond)),
		LogGroupName:  &stream.Group,
		LogStreamName: &stream.Name,
		StartFromHead: aws.Bool(true),
	}, func(resp *cloudwatchlogs.GetLogEventsOutput, lastPage bool) bool {
		if len(resp.Events) != 0 || batches == 0 {
			batches++
		}

		for _, e := range resp.Events {
			events = append(events, &Event{
				OutputLogEvent: e,
				LocalIP:        stream.IP,
			})
		}

		return !lastPage
	})
	if err != nil {
		return batches, nil, fmt.Errorf("getting CWL: %w", err)
	}

	return batches, events, nil
}

func UnmarshalFlow(line string, srcIP string) *Flow {
	split := strings.Fields(line)
	if len(split) != 14 {
		return nil
	}

	version, _ := strconv.ParseFloat(split[0], 64)
	srcPort, _ := strconv.Atoi(split[5])
	dstPort, _ := strconv.Atoi(split[6])
	packets, _ := strconv.ParseInt(split[8], 10, 64)
	bytes, _ := strconv.ParseInt(split[9], 10, 64)
	start, _ := strconv.ParseInt(split[10], 10, 64)
	end, _ := strconv.ParseInt(split[11], 10, 64)
	flow := &Flow{
		Version:   version,
		Account:   split[1],
		Interface: split[2],
		SrcAddr:   split[3],
		DstAddr:   split[4],
		SrcPort:   srcPort,
		DstPort:   dstPort,
		Protocol:  split[7],
		Packets:   packets,
		Bytes:     bytes,
		Start:     time.Unix(start, 0),
		End:       time.Unix(end, 0),
		Action:    split[12],
		Status:    split[13],
		Remote:    isengard.AWSAccount{Name: "unknown"},
	}

	if srcIP == flow.DstAddr {
		flow.RemAddr = flow.SrcAddr
		flow.RemPort = flow.SrcPort

		aws := cidrmap.FindIPowner(net.ParseIP(flow.SrcAddr))
		if aws != nil {
			flow.Remote = aws.Isengard.AWSAccount
		}
	} else if srcIP == flow.SrcAddr {
		flow.RemAddr = flow.DstAddr
		flow.RemPort = flow.DstPort

		aws := cidrmap.FindIPowner(net.ParseIP(flow.DstAddr))
		if aws != nil {
			flow.Remote = aws.Isengard.AWSAccount
		}
	}

	return flow
}
