package funcs

import (
	"crypto/tls"
	"log"
	"net"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/systems/find_ip_owner/pkg/isengard"
	"git.xarth.tv/awsi/eniflowner/pkg/eni"
)

type AggregatedFlow struct {
	AccountID string
	Name      string
	Count     int
	IPPort    string
	Service   string
	LastSeen  time.Time
}

func GetHTTPCert(rem string) string {
	dialer := &net.Dialer{Timeout: time.Second * 3}

	conn, err := tls.DialWithDialer(dialer, "tcp", rem, &tls.Config{InsecureSkipVerify: true})
	if err != nil {
		return ""
	}
	defer conn.Close()

	certs := conn.ConnectionState().PeerCertificates
	for _, cert := range certs {
		if len(cert.DNSNames) == 0 {
			continue
		}

		for _, c := range cert.DNSNames {
			if strings.HasSuffix(c, ".hls.ttvnw.net") {
				return "*.hls.ttvnw.net"
			} else if strings.HasSuffix(c, ".hls.live-video.net") {
				return "*.hls.live-video.net"
			}
		}

		return strings.Join(cert.DNSNames, "; ")
	}

	return "https"
}

// Stat contains the data for a stat tracker.
type Stat struct {
	Count int
	Last  time.Time
}

// Add increases a counter and sets the last seen date.
func (s *Stat) Add(count int, when time.Time) {
	s.Count += count

	if when.After(s.Last) {
		s.Last = when
	}
}

func AggregateEvents(all []*eni.Event, cutset Filter) (int, int, int, map[string]*Stat, map[string]isengard.AWSAccount) {
	var (
		skip, keep, filter int
		agg                = make(map[string]*Stat)
		who                = make(map[string]isengard.AWSAccount)
	)

	for _, event := range all {
		if event == nil || event.Message == nil {
			skip++
			continue
		}

		flow := eni.UnmarshalFlow(*event.Message, event.LocalIP)
		if flow == nil {
			log.Println("Error:", *event.Message)
			skip++
			continue
		} else if flow.Remote.Name == "unknown" {
			skip++
			continue
		} else if cutset.Has(flow.RemPort) {
			filter++
			continue
		}

		keep++

		rem := flow.RemAddr + ":" + strconv.Itoa(flow.RemPort)
		if agg[rem] == nil {
			agg[rem] = &Stat{Count: 1, Last: flow.End}
		} else {
			agg[rem].Add(1, flow.End)
		}

		who[rem] = flow.Remote
	}

	return keep, skip, filter, agg, who
}

func AggregateFlows(agg map[string]*Stat, who map[string]isengard.AWSAccount,
	ipmap map[string]string) ([]*AggregatedFlow, map[string]string) {
	var (
		svc    = make(map[string]string)
		newips = make(map[string]string)
		flows  = []*AggregatedFlow{}
	)

	for rem, stat := range agg {
		flow := &AggregatedFlow{
			AccountID: who[rem].AWSAccountID,
			Name:      who[rem].Name,
			Count:     stat.Count,
			IPPort:    rem,
			Service:   "unknown",
			LastSeen:  stat.Last,
		}

		if flow.Name == "" {
			flow.Name = who[rem].Alias
		}

		if ss, ok := svc[rem]; ok {
			flow.Service = ss
		}

		s := strings.Split(rem, ":")
		if ss, ok := ipmap[s[0]]; ok {
			flow.Service = ss
		}

		if flow.Service == "unknown" {
			switch s[1] {
			case "0":
				flow.Service = "probably icmp"
			case "22":
				flow.Service = "ssh"
			case "443":
				if who[rem].Name != "unknown" {
					// Avoid connecting to external IPs.
					flow.Service = GetHTTPCert(rem)
					if flow.Service == "" {
						flow.Service = "unknown"
						break
					}

					svc[rem] = flow.Service
					newips[s[0]] = flow.Service
				}
			case "80", "8080":
				flow.Service = "http://" + rem
			case "636":
				flow.Service = "ldap"
			case "2003", "8125":
				flow.Service = "statsd"
			case "8140":
				flow.Service = "puppet"
			case "19237", "19238":
				flow.Service = "nagios.xarth.tv"
			case "6379":
				flow.Service = "redis"
			case "5432", "3306":
				flow.Service = "sql"
			case "5439":
				flow.Service = "redshift"
			case "1935":
				flow.Service = "rtsp"
			}
		}

		flows = append(flows, flow)
	}

	return flows, newips
}

// CombineFlows puts together any amount of flow logs.
func CombineFlows(flows ...[]*AggregatedFlow) []*AggregatedFlow {
	agg := make(map[string]*AggregatedFlow)

	for _, aggFlow := range flows {
		for _, flow := range aggFlow {
			if _, ok := agg[flow.IPPort]; !ok {
				agg[flow.IPPort] = flow // Aggregated by destination ip:port.
				continue
			}

			agg[flow.IPPort].Count += flow.Count
			if flow.LastSeen.After(agg[flow.IPPort].LastSeen) {
				agg[flow.IPPort].LastSeen = flow.LastSeen
			}
		}
	}

	retFlows := make([]*AggregatedFlow, len(agg))
	i := 0

	for _, flow := range agg {
		retFlows[i] = flow
		i++
	}

	return retFlows
}
