package report

import (
	"fmt"
	"html/template"
	"os"
	"path/filepath"
	"strings"

	"code.justin.tv/systems/find_ip_owner/pkg/cidrmap"
	"git.xarth.tv/awsi/eniflowner/pkg/funcs"
)

const (
	accountTemplateFile      = "docs/template-account.html"
	accountTemplateIndexFile = "docs/template-account-index.html"
)

// Account contains a report per account.
type Account struct {
	AccountID string
	Name      string
	Meta      *cidrmap.Account
	Ports     map[string]*funcs.Stat
	IPs       map[string]*funcs.Stat
	Svcs      map[string]*funcs.Stat
	Dests     map[string]*funcs.Stat
}

// GetAccounts converts a report into per-Account reports.
func (r *Report) GetAccounts() map[string]*Account {
	accounts := make(map[string]*Account)

	for _, l := range r.Flows {
		if accounts[l.AccountID] == nil {
			accounts[l.AccountID] = &Account{
				AccountID: l.AccountID,
				Name:      l.Name,
				Meta:      cidrmap.Accounts[l.AccountID],
				IPs:       make(map[string]*funcs.Stat),
				Ports:     make(map[string]*funcs.Stat),
				Svcs:      make(map[string]*funcs.Stat),
				Dests:     make(map[string]*funcs.Stat),
			}
		}

		accounts[l.AccountID].addFlow(l)
	}

	return accounts
}

// addFlow adds a new flow to an account.
func (a *Account) addFlow(flow *funcs.AggregatedFlow) {
	ipport := strings.Split(flow.IPPort, ":")

	if a.IPs[ipport[0]] == nil {
		a.IPs[ipport[0]] = &funcs.Stat{Count: flow.Count, Last: flow.LastSeen}
	} else {
		a.IPs[ipport[0]].Add(flow.Count, flow.LastSeen)
	}

	if a.Ports[ipport[1]] == nil {
		a.Ports[ipport[1]] = &funcs.Stat{Count: flow.Count, Last: flow.LastSeen}
	} else {
		a.Ports[ipport[1]].Add(flow.Count, flow.LastSeen)
	}

	if a.Svcs[flow.Service] == nil {
		a.Svcs[flow.Service] = &funcs.Stat{Count: flow.Count, Last: flow.LastSeen}
	} else {
		a.Svcs[flow.Service].Add(flow.Count, flow.LastSeen)
	}

	if a.Dests[flow.IPPort] == nil {
		a.Dests[flow.IPPort] = &funcs.Stat{Count: flow.Count, Last: flow.LastSeen}
	} else {
		a.Dests[flow.IPPort].Add(flow.Count, flow.LastSeen)
	}
}

func (r *Report) WriteAccountHTMLIndex(filename string, a map[string]*Account) error {
	if filename == "" || a == nil {
		return nil
	}

	if err := mkdir(filename); err != nil {
		return err
	}

	htmlFile, err := os.OpenFile(filename+".html", os.O_CREATE|os.O_WRONLY, 0600)
	if err != nil {
		return fmt.Errorf("creating html file: %w", err)
	}
	defer htmlFile.Close()

	return template.Must(template.New(filepath.Base(accountTemplateIndexFile)).
		Funcs(template.FuncMap{
			"accounts": func() map[string]*Account { return a },
			"files":    func() string { return strings.Join(r.Files, ", ") },
		}).
		ParseFiles(accountTemplateIndexFile)).
		Execute(htmlFile, r)
}

// WriteAccountHTML writes an account's html report from a template file.
func (r *Report) WriteAccountHTML(filename string, a *Account) error {
	if filename == "" || r == nil {
		return nil
	}

	if err := mkdir(filename); err != nil {
		return err
	}

	htmlFile, err := os.OpenFile(filename+".html", os.O_CREATE|os.O_WRONLY, 0600)
	if err != nil {
		return fmt.Errorf("creating html file: %w", err)
	}
	defer htmlFile.Close()

	return template.Must(template.New(filepath.Base(accountTemplateFile)).
		Funcs(template.FuncMap{
			"files":  func() string { return strings.Join(r.Files, ", ") },
			"report": func() *Report { return r },
		}).
		ParseFiles(accountTemplateFile)).
		Execute(htmlFile, a)
}
