package report

import (
	"encoding/json"
	"fmt"
	"html/template"
	"os"
	"path/filepath"
	"strings"
	"time"

	"git.xarth.tv/awsi/eniflowner/pkg/funcs"
)

const htmlTemplateFile = "docs/template.html"

type Report struct {
	Title        string
	Start        time.Time
	Age          time.Duration
	Files        []string
	Streams      int
	OldFlows     int
	Events       int
	IPmap        int
	Keep         int
	Skip         int
	Filtered     int
	Destinations int
	NewIPS       int
	Flows        []*funcs.AggregatedFlow
}

// Load loads some json files.
func Load(jsonfiles []string) (*Report, error) {
	report := &Report{Files: jsonfiles}

	for _, j := range jsonfiles {
		oldFlows, err := loadFile(j)
		if err != nil {
			return report, err
		}

		report.Flows = funcs.CombineFlows(report.Flows, oldFlows)
	}

	return report, nil
}

func loadFile(jsonfile string) ([]*funcs.AggregatedFlow, error) {
	r, err := os.Open(jsonfile)
	if err != nil {
		return nil, fmt.Errorf("reading json file: %w", err)
	}

	report := &Report{}
	if err := json.NewDecoder(r).Decode(report); err != nil {
		return nil, fmt.Errorf("decoding json file: %w", err)
	}

	return report.Flows, nil
}

func mkdir(filename string) error {
	err := os.MkdirAll(filepath.Dir(filename), 0775)
	if err != nil {
		return fmt.Errorf("making dirs: %w", err)
	}

	return nil
}

func (r *Report) WriteJSON(filename string) error {
	if filename == "" {
		return nil
	}

	if err := mkdir(filename); err != nil {
		return err
	}

	w, err := os.OpenFile(filename, os.O_CREATE|os.O_WRONLY, 0600)
	if err != nil {
		return fmt.Errorf("opening json file: %w", err)
	}
	defer w.Close()

	if err := json.NewEncoder(w).Encode(r); err != nil {
		return fmt.Errorf("encoding json file: %w", err)
	}

	return nil
}

func (r *Report) WriteCSV(filename string) error {
	if filename == "" {
		return nil
	}

	if err := mkdir(filename); err != nil {
		return err
	}

	w, err := os.OpenFile(filename, os.O_CREATE|os.O_WRONLY, 0600)
	if err != nil {
		return fmt.Errorf("creating csv file: %w", err)
	}
	defer w.Close()

	fmt.Fprintf(w, "Isengard,Name,Flows,IP:Port,Service")

	for _, f := range r.Flows {
		fmt.Fprintf(w, "%s,%s,%d,%s,%s\n", f.AccountID, f.Name, f.Count, f.IPPort, f.Service)
	}

	return nil
}

func (r *Report) WriteHTML(filename string) error {
	if filename == "" {
		return nil
	}

	if err := mkdir(filename); err != nil {
		return err
	}

	w1, err := os.OpenFile(filename+".html", os.O_CREATE|os.O_WRONLY, 0600)
	if err != nil {
		return fmt.Errorf("creating html file: %w", err)
	}
	defer w1.Close()

	w2, err := os.OpenFile(filename+"-baremetal.html", os.O_CREATE|os.O_WRONLY, 0600)
	if err != nil {
		return fmt.Errorf("creating html file: %w", err)
	}
	defer w2.Close()

	r1, r2 := r.splitBaremetal()
	template := template.Must(template.New(filepath.Base(htmlTemplateFile)).ParseFiles(htmlTemplateFile))

	if err := template.Execute(w1, r1); err != nil {
		return err
	}

	return template.Execute(w2, r2)
}

func (r *Report) splitBaremetal() (*Report, *Report) {
	var r1, r2 = &Report{
		Title:        "AWS Only",
		Start:        r.Start,
		Age:          r.Age,
		Streams:      r.Streams,
		OldFlows:     r.OldFlows,
		Events:       r.Events,
		IPmap:        r.IPmap,
		Keep:         r.Keep,
		Skip:         r.Skip,
		Filtered:     r.Filtered,
		Destinations: r.Destinations,
		NewIPS:       r.NewIPS,
		Flows:        []*funcs.AggregatedFlow{},
	}, &Report{
		Title:        "Baremetal",
		Start:        r.Start,
		Age:          r.Age,
		Streams:      r.Streams,
		OldFlows:     r.OldFlows,
		Events:       r.Events,
		IPmap:        r.IPmap,
		Keep:         r.Keep,
		Skip:         r.Skip,
		Filtered:     r.Filtered,
		Destinations: r.Destinations,
		NewIPS:       r.NewIPS,
		Flows:        []*funcs.AggregatedFlow{},
	}

	for _, flow := range r.Flows {
		if strings.EqualFold(flow.AccountID, "BAREMETAL") {
			r2.Flows = append(r2.Flows, flow)
		} else {
			r1.Flows = append(r1.Flows, flow)
		}
	}

	return r1, r2
}
