package main

import (
	"fmt"
	"io/ioutil"
	"log"
	"path/filepath"
	"strings"
)

type User struct {
	Name     string
	Manager  string
	TwitchID string
	AmazonID string
	Director string
}

type Config struct {
	Directors map[string]string
	Users     map[string]*User
	Managers  map[string]string
}

func main() {
	userfiles, _ := filepath.Glob("step3_output/*.csv")
	mngrfiles, _ := filepath.Glob("step4_managers/*.csv")

	c := &Config{
		Directors: map[string]string{
			"dharry":   "Drew Harry",
			"farhadr":  "Farhad Roueintan",
			"gstavely": "Grant Stavley",
			"jacobw":   "Jacob Woodsey",
			"forrjere": "Jeremy Forester",
			"mintonmm": "Mike Minton",
			"browarek": "Sharmeen Browrek",
			"tdrobbo":  "Thomas Robertson",
			"tylerobb": "Tyler Robbins",
			"mhess":    "Martin Hess",
			"trcchan":  "Tracy Chan",
			"wavrmi":   "Michelle Weaver",
			"aragonmi": "Michael Aragon",
			"dugsc":    "Doug Scott",
			"damiburn": "Damian Burns",
			"benestev": "Steve Bene",
			"kaursuni": "Sunita Kaur",
		},
		Users:    getUsers(userfiles),
		Managers: getValues(mngrfiles),
	}

	for k, v := range c.Users {
		c.Managers[k] = v.Manager
	}

	for user, uinfo := range c.Users {
		if _, ok := c.Directors[user]; ok {
			c.Users[user].Director = user
			continue
		}

		director := c.findDirector(uinfo.Manager)
		if _, ok := c.Directors[director]; !ok {
			c.Users[user].Director = "amazon"
		} else {
			c.Users[user].Director = director
		}
	}

	c.printCSV()
}

//nolint:forbidigo
func (c *Config) printCSV() {
	fmt.Println("Full Name,Twitch ID,Amazon ID,Manager,Director")

	for user, uinfo := range c.Users {
		fmt.Printf("%s,%s,%s,%s,%s\n", uinfo.Name, uinfo.TwitchID, user, uinfo.Manager, uinfo.Director)
	}
}

func (c *Config) findDirector(manager string) (director string) {
	if _, ok := c.Directors[manager]; ok {
		return manager
	}

	if mm := c.Managers[manager]; mm != "" {
		return c.findDirector(mm)
	}

	return manager
}

func getUsers(files []string) map[string]*User {
	values := make(map[string]*User, len(files))

	for _, f := range files {
		data, err := ioutil.ReadFile(f)
		if err != nil {
			log.Fatal(err)
		}

		username := strings.TrimSuffix(filepath.Base(f), ".csv")
		split := strings.Split(strings.TrimSpace(string(data)), ",")
		values[username] = &User{
			Name:     split[3],
			Manager:  split[1],
			TwitchID: split[2],
			AmazonID: split[0],
		}
	}

	return values
}

func getValues(files []string) map[string]string {
	values := make(map[string]string, len(files))

	for _, f := range files {
		data, err := ioutil.ReadFile(f)
		if err != nil {
			log.Fatal(err)
		}

		username := strings.TrimSuffix(filepath.Base(f), ".csv")
		values[username] = strings.Split(strings.TrimSpace(string(data)), ",")[1] // manager
	}

	return values
}
