#!/bin/bash
# This script builds a Go docker image and optionally pushes it to Artifactory.

set -e -o pipefail

source "$(dirname $(realpath $0))/env.sh"

# DEPLOY being set to "deploy" triggers docker push
DEPLOY=$1
# VERSION must be a version:sha:baseos combo from getlatest or getneeded scripts.
VERSION=$2
# GOLANGCI_LINT_VERSION sets a version for this tool. Default 'latest' provided.
GOLANGCI_LINT_VERSION=$3
[ "$GOLANGCI_LINT_VERSION" != "" ] || GOLANGCI_LINT_VERSION=latest

v=$(echo "${VERSION}" | cut -d: -f1)
hash=$(echo "${VERSION}" | cut -d: -f2)
baseos=$(echo "${VERSION}" | cut -d: -f3)
image="${REGISTRY}/${NAMESPACE}/${baseos}/${v}:latest"
imagetag="${REGISTRY}/${NAMESPACE}/${baseos}/${v}:$(date +%Y%m%dT%H%M%S)"

[ -n "$GIT_COMMIT" ] || GIT_COMMIT=$(git rev-parse HEAD 2>/dev/null)

if [ -z "$v" ] || [ -z "${hash}" ] || [ -z "$baseos" ]; then
  echo "==> Pass a version:sha:baseos as second arg, ex: go1.12.1:abcdef:bionic" >&2
  exit 1
fi

echo "==> Building: ${image}"
if ! docker build --pull --no-cache . --tag "${image}" \
  --build-arg "GOLANGCI_LINT_VERSION=${GOLANGCI_LINT_VERSION}" \
  --build-arg "GOLANG_VERSION=${v}" \
  --build-arg "GOLANG_SRC_SHA256=${hash}" \
  --build-arg "LINUX_DIST=${baseos}" \
  --build-arg "REGISTRY=${REGISTRY}" \
  --build-arg "GOLANG_SRC_URL=https://golang.org/dl/${v}.linux-amd64.tar.gz" \
  --build-arg "BUILD_DATE=$(date -u '+%Y/%m/%d %H:%M:%S %Z')" \
  --build-arg "COMMIT=${GIT_COMMIT}";
then
  echo "==> Docker build failed for ${baseos}/${v}" >&2
  exit 1
fi

echo "==> Testing Built Image: ${baseos}/${v}"
if ! docker run -v "$(pwd)/test:/test" "${image}" /test/test;
then
  echo "==> Docker test failed for ${baseos}/${v}" >&2
  exit 1
fi

# Run this here to make sure any error is caught.
echo "==> Tagging Built Image: ${baseos}/${v}: ${image} => ${imagetag}"
docker tag "${image}" "${imagetag}"

# $DEPLOY is set from $1 and comes from Jenkinsfile.
if [ "$DEPLOY" != "deploy" ]; then
  echo "==> Skipping Deploy: ${baseos}/${v}"
  exit 0
fi

echo "==> Pushing Built Image: ${baseos}/${v}: ${image}"
if ! docker push "${image}";
then
  echo "==> Docker push failed for ${baseos}/${v}" >&2
  exit 1
fi

echo "==> Pushing Built Image: ${baseos}/${v}: ${imagetag}"
if ! docker push "${imagetag}";
then
  echo "==> Docker push failed for ${baseos}/${v} (date tag)" >&2
  exit 1
fi
