# Auto Delegator: twitch-a2z-com

This repo contains a Go Lambda that processes Route53 Public Hosted Zone delegation requests.
The code was written to delegate twitch.a2z.com sub zone, but is fully adaptable to any zone.

## Description

Provided are two cloud formation stacks that make delegating sub-zones an automatic process
for all accounts in an AWS Org. The primary stack deploys a Go Lambda that handles CRUD
operations for a sub-zone delegation on a "primary" zone (like twitch.a2z.com).
The delegation is initiated with another (consumer) CloudFormation stack that uses a
custom resource to notify an SNS topic. The SNS topic is org-writable and designed specifically
for a custom resource.

Either stack can be deployed multiple times in the same account, and/or in different regions.
This allows serving many zones from one account or delegating many sub-zones into a single account.
The code is easily adaptable to other zones, but an [example](cloudformation/delegation.yaml)
is provided for `twitch.a2z.com` (production) and `testing.x2y.dev` (fake test zone).

[flow-chart-link]: # "https://lucid.app/lucidchart/435f5ecc-ec4e-4abd-a928-bcd69ace5e37/edit#?folder_id=home&browser=icon"
[![delegation-flow](https://git.xarth.tv/pages/awsi/twitch-a2z-com/delegation-flow.svg "Public Zone Delegation Flow")](https://git.xarth.tv/pages/awsi/twitch-a2z-com/delegation-flow.svg)

## Dangling Delegation Checker

Another feature of the delegator CloudFormation stack is the dangling delegation checker.
This system is composed of two small lambda apps and an SQS queue. The first lambda is
triggered by a CloudWatch Event. It collects all NS records from your main zone and sends
them to an SQS queue. The second lambda processes the SQS queue by validating each name
server for the delegated sub-zone. If all four name servers fail to answer an SOA request,
the delegation is immediately deleted.

### Slack Integration

This stack comes with a "mandatory" Slack integration. This integration sends all
name server errors and all Route53 RecordSet changes to a Slack Channel. **You must
run this stack in us-east-1**; assuming you don't remove the Slack-portion of the
CFN Stack. At a minimum, add your Slack WebHook URL to SSM parameter store in
us-east-1 as `/slack/webhook.url`, and set the stack parameter for your channel.

## Deployment

If you own a zone that you want to allow other twitch users to delegate themselves, this is how:
Deploy the delegator.yaml stack to your account using the commands you see below.
Replace the hosted zone ID with your hosted zone ID.

Once you've deployed the delegator stack, a consumer will deploy the delegation stack. First,
you must make a CloudFormation template specific to your delegator stack by using the example at
[cloudformation/delegation.yaml](cloudformation/delegation.yaml).
Copy the example file, and add a new `DelegationMap` value for your zone name.
The `ServiceToken` is the SNS topic created by the delegator stack; it follows this format:
`arn:aws:sns:us-east-1:${AccountID}:auto-delegator-cfn-${ZoneID}`. Add your zone to the
`MainZoneName` Parameter `AllowedValues` list.

Give the pre-filled delegation stack to your consumers.
They need to create a sub-zone of your zone and then deploy the stack.

### Dev-Account Example

These commands build the app, and deploy it to the testing account `twitcha2z-backup`.
You must run `mwinit` first and add profile/credentials.

#### twitch.a2z.com (prod copy)

```
GOOS=linux GOARCH=amd64 go build ./cmd/lambda && zip -m9 ./lambda.zip ./lambda && \
aws --profile=twitcha2z-backup --region=us-east-1 cloudformation package \
    --template-file cloudformation/delegator.yaml \
    --output-template-file delegator-deploy.yaml \
    --s3-bucket twitch-a2z-com-delegator-backup-us-east-1 && \
aws --profile=twitcha2z-backup --region=us-east-1 cloudformation deploy \
    --template-file delegator-deploy.yaml \
    --stack-name twitch-a2z-com-testing \
    --capabilities CAPABILITY_IAM \
    --parameter-overrides HostedZoneId=Z0896921FULGP10H8IN9 CheckFrequency=20 SafeMode=false SlackChan=brandon-cool
```

#### testing.x2y.dev (fake)

```
GOOS=linux GOARCH=amd64 go build ./cmd/lambda && zip -m9 ./lambda.zip ./lambda && \
aws --profile=twitcha2z-backup --region=us-east-1 cloudformation package \
    --template-file cloudformation/delegator.yaml \
    --output-template-file delegator-deploy.yaml \
    --s3-bucket twitch-a2z-com-delegator-backup-us-east-1 && \
aws --profile=twitcha2z-backup --region=us-east-1 cloudformation deploy \
    --template-file delegator-deploy.yaml \
    --stack-name testing-x2y-dev-delegator \
    --capabilities CAPABILITY_IAM \
    --parameter-overrides HostedZoneId=Z03963481Y0GZ5RNBTT16 CheckFrequency=2 SafeMode=true SlackChan=brandon-cool
```

#### Consumer URLs

These files must be uploaded. See **Notes** below.

-   `http://s3.amazonaws.com/auto-delegation-testing-x2y-dev-delegator/delegation.yaml`
-   `http://s3.amazonaws.com/auto-delegation-testing-x2y-dev-delegator/delegation-with-zone.yaml`

### Production-Account Example

These commands build the app, and deploy it to the production AWS account `twitcha2z`.
You must run `mwinit` first and add profile/credentials.

#### twitch.a2z.com (production)

You must add the SSM parameter `/slack/webhook.url` before deploying this, even if it's blank.
If you add it blank, you must re-deploy after you correct it.

```
GOOS=linux GOARCH=amd64 go build ./cmd/lambda && zip -m9 ./lambda.zip ./lambda && \
aws --profile=twitcha2z --region=us-east-1 cloudformation package \
    --template-file cloudformation/delegator.yaml \
    --output-template-file delegator-deploy.yaml \
    --s3-bucket twitch-a2zdns-cf-us-east-1 && \
aws --profile=twitcha2z --region=us-east-1 cloudformation deploy \
    --template-file delegator-deploy.yaml \
    --stack-name twitch-a2z-com-delegator \
    --capabilities CAPABILITY_IAM \
    --parameter-overrides CheckFrequency=2 SlackChan=C01H71D4RCK
```

#### Consumer URLs

-   `http://s3.amazonaws.com/auto-delegation-twitch-a2z-com-delegator/delegation.yaml`
-   `http://s3.amazonaws.com/auto-delegation-twitch-a2z-com-delegator/delegation-with-zone.yaml`

These files must be uploaded, like this:

```
aws --profile=twitcha2z --region=us-east-1 s3 cp cloudformation/delegation.yaml s3://auto-delegation-twitch-a2z-com-delegator/
aws --profile=twitcha2z --region=us-east-1 s3 cp cloudformation/delegation-with-zone.yaml s3://auto-delegation-twitch-a2z-com-delegator/
```

## Create - Explained

The consumer stack creates an IAM role that the delegator account assumes, to locate the Name and Nameservers for the Zone ID.
The only other resource it creates is an SNS topic notification into the delegator account.

The Go Lambda can only be invoked by an AWS Orgs member, so this provides our initial security.
-   The app accepts **only a zone ID** as input from SNS.
-   The account ID is parsed from the trustable AWS data.
-   A pre-known role is assumed in the "trusted" account ID.
-   The role is used to retrieve the public hosted zone's name servers for the provided zone ID.
-   Checks the main zone for an existing delegation with the same name.
-   Checks for an existing sub-sub-delegation (which the new delegation would break).
-   Makes sure the new delegation is in the main zone, ie. `service.twitch.a2z.com` _is in_ `twitch.a2z.com`.

Once the the request is validated, the lambda writes a json blob into s3 with a `requested` status.
Then the delegation `NS RecordSet` is created. Upon success, the s3 blob is updated with a `granted` status.
This json blob is used to authenticate ownership. The saved account ID, zone ID and stack ID must match to make changes.

Any error in the process is returned to the caller, and the s3 blob is not set to `granted`.

## Delete - Explained

Delete is triggered when a consumer stack is deleted or updated. The update operation runs
`delete` on the old zone ID value and then runs `create` (shown above) on the new zone ID value.

When the stack is deleted the following things happen. Keep in mind, we're working with trusted data from
AWS. Anyone with "read" access on an account can trigger a message to this SNS topic to initiate a delete.
Keep that in mind when giving out access.
-   The consumer stack sends a notification to the SNS topic.
-   The Go Lambda uses the trusted AWS data `oldPhysicalId` to find the sub-zone in the main zone.
-   Reads in the json blob stored in s3 for this account ID + zone ID + stack ID combination.
-   Verifies ownership, and that it's in a `granted` state (to avoid modifying something inadvertently).
-   Deletes the NS records from the main zone.
-   Writes a `deleted` status to the json blob in s3.

## Mocks & Tests

This application has many tests. Please keep adding more!

To make mocks, add the magic `//go:generate mockgen` to the top of the interface file and run this:

```
go generate ./...
```

## Notes

In the backup a2z account, I've manually:
-   Created a few route53 zones: `testing.x2y.dev` and some sub zones.
-   Created SSM parameter store value: `/slack/webhook.url` with our slack webhook as a text value.
-   Made a direct copy of `twitch.a2z.com` from production into this account.
    -   351 NS delegations, 355 record sets.
    -   TODO: automate this copy process into a canary account.
-   Created s3 bucket to hold the cfn lambdas: `twitch-a2z-com-delegator-backup`
-   Deployed the [delegator stack](cloudformation/delegator.yaml).
-   Copied the [delegation stacks](cloudformation/) into the public s3 bucket:
-   `aws --profile=twitcha2z-backup --region=us-east-1 s3 cp cloudformation/delegation.yaml s3://auto-delegation-testing-x2y-dev-delegator/`
-   `aws --profile=twitcha2z-backup --region=us-east-1 s3 cp cloudformation/delegation-with-zone.yaml s3://auto-delegation-testing-x2y-dev-delegator/`
