// Package backup downloads a zone's NS records and saves them to S3.
// Designed to be run as a lambda at an interval.
package backup

import (
	"context"
	"fmt"
	"log"
	"os"
	"path"
	"time"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"code.justin.tv/awsi/twitch-a2z-com/pkg/storage"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/service/route53"
	"github.com/aws/aws-sdk-go/service/s3"
)

// Config is the input data needed to run this lambda handler.
type Config struct {
	R53 *delegate.Delegate
	S3  *storage.Storage
}

// New returns a config generated from lambda environment variables.
func New(sess client.ConfigProvider, awsc *aws.Config) *Config {
	return &Config{
		R53: &delegate.Delegate{
			Svc:    route53.New(sess, awsc),
			ZoneID: os.Getenv("ZONEID"), // not optional.
		},
		S3: &storage.Storage{
			Svc:    s3.New(sess, awsc),
			Prefix: os.Getenv("S3_KEYPREFIX"), // used as a path prefix in the s3 bucket.
			Bucket: os.Getenv("S3_BUCKET"),    // not optional.
		},
	}
}

// Handler runs from a lambda and backs up a zone.
func (c *Config) Handler(ctx context.Context) error {
	delegations, err := c.R53.GetResourceRecords(ctx,
		&route53.ListResourceRecordSetsInput{HostedZoneId: &c.R53.ZoneID}, true)
	if err != nil {
		log.Println("[R53 ERROR]", err)
		return fmt.Errorf("error scanning zone %s: %w", c.R53.ZoneID, err)
	}

	now := time.Now()
	s3key := now.Format("2006/Jan/02/") + c.R53.ZoneID + now.Format("-15:04")

	err = c.S3.SaveRaw(ctx, c.delegations2batchchange(delegations), s3key)
	if err != nil {
		log.Println("[S3 ERROR]", err)
		return fmt.Errorf("saving backup to S3 %s: %w", s3key, err)
	}

	log.Printf("[INFO] Saved %d records to s3://%s", len(delegations),
		path.Join(c.S3.Bucket, c.S3.Prefix, s3key))

	return nil
}

// delegations2batchchange converts a delegation set into a change batch.
// This allows easier recovery by having the data in an importable format.
func (c *Config) delegations2batchchange(delegations []*delegate.Delegation) interface{} {
	cb := &route53.ChangeResourceRecordSetsInput{
		HostedZoneId: &c.R53.ZoneID,
		ChangeBatch:  &route53.ChangeBatch{Changes: []*route53.Change{}},
	}

	for _, delegation := range delegations {
		cb.ChangeBatch.Changes = append(cb.ChangeBatch.Changes, &route53.Change{
			Action: aws.String(route53.ChangeActionCreate),
			ResourceRecordSet: &route53.ResourceRecordSet{
				Name:            &delegation.Subzone,
				ResourceRecords: delegation.Nameservers,
				TTL:             delegation.TTL,
				Type:            delegation.Type,
			},
		})
	}

	return cb
}
