package delegate

import (
	"fmt"
	"strings"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/route53"
)

/* These procedures are used when creating (and updating) a new delegation. */

// Create adds the NS records for a subzone to our main zone.
func (d *Delegate) Create(ctx aws.Context, request *Delegation) error {
	for _, ns := range request.Nameservers {
		if strings.Contains(*ns.Value, ".awsdns-00.") { // private DNS servers contain this string.
			return fmt.Errorf("%w: %s/%s", ErrHostedZoneNotPublic, request.ZoneID, request.Subzone)
		}
	}

	_, err := d.Svc.ChangeResourceRecordSetsWithContext(ctx, &route53.ChangeResourceRecordSetsInput{
		HostedZoneId: aws.String(d.ZoneID),
		ChangeBatch: &route53.ChangeBatch{
			Changes: []*route53.Change{{
				Action: aws.String(route53.ChangeActionCreate),
				ResourceRecordSet: &route53.ResourceRecordSet{
					Name:            aws.String(request.Subzone),
					ResourceRecords: request.Nameservers,
					TTL:             aws.Int64(d.TTL),
					Type:            aws.String(route53.RRTypeNs),
				},
			}},
			Comment: aws.String("AccountID: " + request.AccountID + ", ZoneID: " + request.ZoneID),
		},
	})
	if err != nil {
		return fmt.Errorf("ChangeResourceRecordSetsWithContext: %w", err)
	}

	return nil
}

// GetZone uses the provided svc Delegator to lookup a zone with default or assumed credentials.
// This is used to lookup the main zone [from our account] and delegated zones [from remote accounts].
func (d *Delegate) GetZone(ctx aws.Context, accountID, zoneID string, svc Delegator) (*Delegation, error) {
	if svc == nil {
		svc = d.Svc
	}

	zone, err := svc.GetHostedZoneWithContext(ctx, &route53.GetHostedZoneInput{Id: aws.String(zoneID)})
	if err != nil {
		return nil, fmt.Errorf("GetHostedZoneWithContext: %w", err)
	}

	if *zone.HostedZone.Config.PrivateZone {
		return nil, fmt.Errorf("%w: %s/%s", ErrHostedZoneNotPublic, zoneID, *zone.HostedZone.Name)
	}

	nss := NameServers{}
	for _, ns := range zone.DelegationSet.NameServers { // make sure each ns ends with a dot.
		nss = append(nss, &route53.ResourceRecord{Value: aws.String(strings.TrimSuffix(*ns, ".") + ".")})
	}

	return &Delegation{
		AccountID:   accountID,
		Subzone:     *zone.HostedZone.Name,
		Nameservers: nss,
		ZoneID:      zoneID,
		Type:        aws.String(route53.RRTypeNs),
	}, nil
}

// CheckNewZone makes sure a new delegation wont stomp on an existing delegation.
// It does this by pulling all the existing NS records and comparing them to the request.
// Also makes sure the requested subzone can exist in the requested zone (it's actually a sub zone).
func (d *Delegate) CheckNewZone(ctx aws.Context, delegation *Delegation) error {
	// The subzone must be in the main zone, and the subzone must not equal the main zone.
	if !strings.HasSuffix(delegation.Subzone, d.ZoneName) || strings.EqualFold(delegation.Subzone, d.ZoneName) {
		return fmt.Errorf("%s: %w %s", delegation.Subzone, ErrSubZoneNotInZone, d.ZoneName)
	}

	rr, err := d.GetResourceRecords(ctx, &route53.ListResourceRecordSetsInput{HostedZoneId: &d.ZoneID}, true)
	if err != nil {
		return fmt.Errorf("ListResourceRecordSetsWithContext: %w", err)
	}

	for _, record := range rr {
		// Do not allow duplicate delegations.
		if strings.EqualFold(delegation.Subzone, record.Subzone) {
			return fmt.Errorf("%s: %w", delegation.Subzone, ErrDelegationExists)
		}

		// Do not allow delegating a zone with an existing subzone.
		if strings.HasSuffix(record.Subzone, "."+delegation.Subzone) {
			return fmt.Errorf("%s: %w: %s", delegation.Subzone, ErrConflictingRecord, record.Subzone)
		}
	}

	return nil
}

// GetResourceRecords deals with pagination and gets _all_ the records.
func (d *Delegate) GetResourceRecords(ctx aws.Context,
	input *route53.ListResourceRecordSetsInput, all bool) ([]*Delegation, error) {
	delegations := []*Delegation{}
	input.MaxItems = aws.String("300")

	err := d.Svc.ListResourceRecordSetsPagesWithContext(ctx, input,
		func(rr *route53.ListResourceRecordSetsOutput, lastPage bool) bool {
			for _, record := range rr.ResourceRecordSets {
				// Append all NS records that are not for the main zone.
				if strings.EqualFold(d.ZoneName, *record.Name) {
					continue
				}

				if *record.Type == route53.RRTypeNs || all {
					delegations = append(delegations, &Delegation{
						Subzone:     *record.Name,
						Nameservers: record.ResourceRecords,
						TTL:         record.TTL,
						Type:        record.Type,
					})
				}
			}

			return !lastPage
		},
	)
	if err != nil {
		return nil, fmt.Errorf("ListResourceRecordSetsWithContext: %w", err)
	}

	return delegations, nil
}
