package delegate_test

import (
	"testing"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/route53"
	gomock "github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
)

func TestCreate(t *testing.T) {
	t.Parallel()
	assert := assert.New(t)

	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	d, delegator := testNewConfig(mockCtrl)
	ctx := aws.BackgroundContext()
	req := &delegate.Delegation{
		ZoneID:    "JH32G4H3H234H34GH",
		Subzone:   "suuuuub.zoooon",
		AccountID: "312434324234",
		Nameservers: []*route53.ResourceRecord{
			{Value: aws.String("ns1.zone")},
			{Value: aws.String("ns2.zone")},
			{Value: aws.String("ns3.zone")},
		},
	}

	// Make sure all the data passed into the Record Creation are accurate.
	delegator.EXPECT().ChangeResourceRecordSetsWithContext(ctx, gomock.Any(), []request.Option{}).Do(
		func(_ aws.Context, obj *route53.ChangeResourceRecordSetsInput, _ ...request.Option) {
			assert.Equal(d.ZoneID, *obj.HostedZoneId)
			assert.Equal(route53.ChangeActionCreate, *obj.ChangeBatch.Changes[0].Action)
			assert.Equal(d.TTL, *obj.ChangeBatch.Changes[0].ResourceRecordSet.TTL)
			assert.Equal(route53.RRTypeNs, *obj.ChangeBatch.Changes[0].ResourceRecordSet.Type)
			assert.Equal(req.Subzone, *obj.ChangeBatch.Changes[0].ResourceRecordSet.Name)
			assert.Contains(*obj.ChangeBatch.Comment, req.ZoneID)
			assert.Contains(*obj.ChangeBatch.Comment, req.AccountID)
		},
	).Return(nil, nil)
	assert.Nil(d.Create(ctx, req))

	delegator.EXPECT().ChangeResourceRecordSetsWithContext(ctx, gomock.Any(), []request.Option{}).Return(nil, errTest)
	assert.ErrorIs(d.Create(ctx, req), errTest, "the provided error must be returned.")

	// Now test some error conditions.
	req.Nameservers = append(req.Nameservers, &route53.ResourceRecord{Value: aws.String("ns-101.awsdns-00.aws.net")})
	assert.ErrorIs(d.Create(ctx, req), delegate.ErrHostedZoneNotPublic,
		"An NS with a value containing awsdns-00 must be flagged as non-public.")
}

func TestGetZone(t *testing.T) {
	t.Parallel()
	assert := assert.New(t)

	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	d, delegator := testNewConfig(mockCtrl)
	ctx := aws.BackgroundContext()
	req := &delegate.Delegation{
		ZoneID:    "JH32G4H3H234H34GH",
		Subzone:   "suuuuub.zoooon.com",
		AccountID: "312434324234",
		Nameservers: []*route53.ResourceRecord{
			{Value: aws.String("ns1.zone.")},
			{Value: aws.String("ns2.zone.")},
			{Value: aws.String("ns3.zone.")},
		},
	}
	input := &route53.GetHostedZoneInput{Id: aws.String(req.ZoneID)}
	fake := &route53.GetHostedZoneOutput{
		DelegationSet: &route53.DelegationSet{NameServers: req.Nameservers.Slice()},
		HostedZone: &route53.HostedZone{
			Name:   aws.String("zoooon.com"),
			Config: &route53.HostedZoneConfig{PrivateZone: aws.Bool(true)},
		},
	}

	delegator.EXPECT().GetHostedZoneWithContext(ctx, input, []request.Option{}).Return(nil, errTest)
	od, err := d.GetZone(ctx, req.AccountID, req.ZoneID, delegator)
	assert.Nil(od)
	assert.ErrorIs(err, errTest, "the error must be returned")

	delegator.EXPECT().GetHostedZoneWithContext(ctx, input, []request.Option{}).Return(fake, nil)
	od, err = d.GetZone(ctx, req.AccountID, req.ZoneID, delegator)
	assert.Nil(od)
	assert.ErrorIs(err, delegate.ErrHostedZoneNotPublic, "the zone is marked private, this should fail")

	fake.HostedZone.Config.PrivateZone = aws.Bool(false)
	delegator.EXPECT().GetHostedZoneWithContext(ctx, input, []request.Option{}).Return(fake, nil)
	od, err = d.GetZone(ctx, req.AccountID, req.ZoneID, delegator)
	assert.Nil(err, "this valid request must not produce an error")
	assert.Equal(req.AccountID, od.AccountID)
	assert.Equal(req.ZoneID, od.ZoneID)
	assert.Equal(*fake.HostedZone.Name, od.Subzone)
	assert.Equal(req.Nameservers, od.Nameservers)
}

func TestCheckNewZone(t *testing.T) { // nolint: funlen
	t.Parallel()
	assert := assert.New(t)

	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	d, delegator := testNewConfig(mockCtrl)
	ctx := aws.BackgroundContext()
	req := &delegate.Delegation{
		ZoneID:    "JH32G4H3H234H34GH",
		Subzone:   "suuuuub.zoooon.main.WRONG",
		AccountID: "312434324234",
		Nameservers: []*route53.ResourceRecord{
			{Value: aws.String("ns1.zone")},
			{Value: aws.String("ns2.zone")},
			{Value: aws.String("ns3.zone")},
		},
	}
	input := &route53.ListResourceRecordSetsInput{HostedZoneId: aws.String(d.ZoneID), MaxItems: aws.String("300")}
	returns := &route53.ListResourceRecordSetsOutput{
		ResourceRecordSets: []*route53.ResourceRecordSet{
			// This is what our zone 'main.zone.' looks like.
			{Name: aws.String("main.zone."), Type: aws.String("MX")},
			{Name: aws.String("main.zone."), Type: aws.String("A")},
			{Name: aws.String("main.zone."), Type: aws.String("NS")},
			{Name: aws.String("ns1.main.zone."), Type: aws.String("NS")},
			{Name: aws.String("www.main.zone."), Type: aws.String("CNAME")},
			{Name: aws.String("suuuuub.zoooon.main.zone."), Type: aws.String("NS")},
		},
	}
	do := func(ctx aws.Context, input *route53.ListResourceRecordSetsInput,
		fn func(*route53.ListResourceRecordSetsOutput, bool) bool, opts ...request.Option) {
		fn(returns, false)
	}

	// This first test has a subzone that's not in the main zone.
	assert.ErrorIs(d.CheckNewZone(ctx, req), delegate.ErrSubZoneNotInZone, "these zones do not match")

	req.Subzone = "main.zone." // don't give out our main zone.
	assert.ErrorIs(d.CheckNewZone(ctx, req), delegate.ErrSubZoneNotInZone, "you can't have that name!")

	req.Subzone = "suuuuub.zoooon.main.zone." // it exists already
	///
	delegator.EXPECT().ListResourceRecordSetsPagesWithContext(ctx, input, gomock.Any()).Return(errTest).Do(do)
	assert.ErrorIs(d.CheckNewZone(ctx, req), errTest, "this error should be passed through")

	delegator.EXPECT().ListResourceRecordSetsPagesWithContext(ctx, input, gomock.Any()).Return(nil).Do(do)
	assert.ErrorIs(d.CheckNewZone(ctx, req), delegate.ErrDelegationExists, "this name is already in our zone")

	req.Subzone = "zoooon.main.zone." // has a sub record already
	///
	delegator.EXPECT().ListResourceRecordSetsPagesWithContext(ctx, input, gomock.Any()).Return(nil).Do(do)
	assert.ErrorIs(d.CheckNewZone(ctx, req), delegate.ErrConflictingRecord, "this record has a conflict and must error")

	req.Subzone = "www.main.zone." // exists as CNAME
	///
	delegator.EXPECT().ListResourceRecordSetsPagesWithContext(ctx, input, gomock.Any()).Return(nil).Do(do)
	assert.ErrorIs(d.CheckNewZone(ctx, req), delegate.ErrDelegationExists, "do not stomp CNAMEs")

	req.Subzone = "this.is.unused.main.zone." // brand spankin new
	///
	delegator.EXPECT().ListResourceRecordSetsPagesWithContext(ctx, input, gomock.Any()).Return(nil).Do(do)
	assert.Nil(d.CheckNewZone(ctx, req), "this is a valid request")
}
