package delegate

//nolint:lll
//go:generate mockgen -destination=../mocks/mock_delegator.go -package=mocks code.justin.tv/awsi/twitch-a2z-com/pkg/delegate Delegator

import (
	"fmt"
	"strings"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/route53"
)

// Errors.
var (
	ErrHostedZoneNotPublic = fmt.Errorf("the hosted zone is not public")
	ErrSubZoneNotInZone    = fmt.Errorf("subzone is not in zone")
	ErrDelegationMissing   = fmt.Errorf("delegation record not found")
	ErrDelegationExists    = fmt.Errorf("delegation already exists")
	ErrConflictingRecord   = fmt.Errorf("conflicting record exists")
)

// DefaultTTL is 2 days + 1 minute TTL for new NS records. DO NOT CHANGE: this is used as a record "key".
const DefaultTTL = int64(60*60*24*2 + 60)

// Delegate is our main data struct with immutable configurations.
// ZoneName and ZoneID represents our main zone, like twitch.a2z.com.
type Delegate struct {
	ZoneName string
	ZoneID   string
	TTL      int64
	Svc      Delegator
}

// Delegation contains the info needed to create, update or remove a delegation.
// This struct is used as an input and an output in all the procedures.
type Delegation struct {
	ZoneID      string      `json:"zone_id,omitempty"`
	Subzone     string      `json:"subzone"`
	AccountID   string      `json:"account_id,omitempty"`
	Nameservers NameServers `json:"nameservers"`
	TTL         *int64      `json:"ttl"`
	Type        *string     `json:"type"`
}

// NameServers just allows us to make a string.Stringer.
type NameServers []*route53.ResourceRecord

// Delegator allows mocking the calls to route53.
type Delegator interface {
	GetHostedZoneWithContext(ctx aws.Context, input *route53.GetHostedZoneInput,
		opts ...request.Option) (*route53.GetHostedZoneOutput, error)
	ChangeResourceRecordSetsWithContext(ctx aws.Context, input *route53.ChangeResourceRecordSetsInput,
		opts ...request.Option) (*route53.ChangeResourceRecordSetsOutput, error)
	ListResourceRecordSetsPagesWithContext(ctx aws.Context, input *route53.ListResourceRecordSetsInput,
		fn func(*route53.ListResourceRecordSetsOutput, bool) bool, opts ...request.Option) error
}

// SaveOwnZone makes sure we have our main zone name. Has to be looked-up from the zone ID.
func (d *Delegate) SaveOwnZone(ctx aws.Context) error {
	if d.ZoneName != "" {
		return nil
	}

	z, err := d.GetZone(ctx, "", d.ZoneID, nil)
	if err == nil {
		d.ZoneName = z.Subzone
	}

	return err
}

// String turns a name server list into a printable string.
func (nss NameServers) String() string {
	var display string

	for i, ns := range nss {
		if i > 0 {
			display += ", "
		}

		display += *ns.Value
	}

	return display
}

// Slice turns a name server list into a route53.DelegationSet object.
func (nss NameServers) Slice() []*string {
	s := []*string{}

	for _, ns := range nss {
		s = append(s, ns.Value)
	}

	return s
}

// Contains checks if a name server exists in our list.
func (nss NameServers) Contains(nameserver string) bool {
	for _, ns := range nss {
		if strings.EqualFold(strings.TrimSuffix(*ns.Value, "."), strings.TrimSuffix(nameserver, ".")) {
			return true
		}
	}

	return false
}
