package delegate_test

import (
	"fmt"
	"testing"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"code.justin.tv/awsi/twitch-a2z-com/pkg/mocks"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/route53"
	gomock "github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
)

var errTest = fmt.Errorf("this is a test error")

// Returns a main Delegate struct with some data and a mocked route53 interface.
func testNewConfig(mockCtrl *gomock.Controller) (*delegate.Delegate, *mocks.MockDelegator) {
	delegator := mocks.NewMockDelegator(mockCtrl)

	return &delegate.Delegate{
		ZoneName: "main.zone.",
		ZoneID:   "M41N-V3RYC007Z0N31D",
		Svc:      delegator,
	}, delegator
}

func TestSaveOwnZone(t *testing.T) {
	t.Parallel()
	assert := assert.New(t)

	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	d, delegator := testNewConfig(mockCtrl)
	ctx := aws.BackgroundContext()

	// This is the data we expect to be passed into GetHostedZoneWithContext
	expected := &route53.GetHostedZoneInput{Id: aws.String(d.ZoneID)}
	returnValue := &route53.GetHostedZoneOutput{
		DelegationSet: &route53.DelegationSet{NameServers: []*string{aws.String("name.server.1")}},
		HostedZone: &route53.HostedZone{
			Name:   aws.String("main.zone."),
			Config: &route53.HostedZoneConfig{PrivateZone: aws.Bool(false)},
		},
	}

	d.ZoneName = ""

	delegator.EXPECT().GetHostedZoneWithContext(ctx, expected, []request.Option{}).Return(returnValue, nil)
	assert.Nil(d.SaveOwnZone(ctx), "this must not return an error")
	assert.EqualValues("main.zone.", d.ZoneName, "the zone name must be updated with the reply")

	d.ZoneName = "" // once more with an error

	delegator.EXPECT().GetHostedZoneWithContext(ctx, expected, []request.Option{}).Return(nil, errTest)
	assert.ErrorIs(d.SaveOwnZone(ctx), errTest, "the error retruned must be passed through")

	d.ZoneName = "zone.exists" // once more with an existing zone.

	assert.Nil(d.SaveOwnZone(ctx), "this must not return an error")
	assert.EqualValues("zone.exists", d.ZoneName, "the zone name must not be changed")
}

func TestGetResourceRecords(t *testing.T) {
	t.Parallel()
	assert := assert.New(t)
	ctx := aws.BackgroundContext()

	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	returns := &route53.ListResourceRecordSetsOutput{
		ResourceRecordSets: []*route53.ResourceRecordSet{{
			Type: aws.String(route53.RRTypeNs),
			Name: aws.String("subzoon.main.zone"),
			TTL:  aws.Int64(999),
			ResourceRecords: []*route53.ResourceRecord{
				{Value: aws.String("ns1")},
				{Value: aws.String("ns2")},
				{Value: aws.String("ns3")},
				{Value: aws.String("ns4")},
			},
		}},
	}
	d, r53 := testNewConfig(mockCtrl)
	do := func(ctx aws.Context, input *route53.ListResourceRecordSetsInput,
		fn func(*route53.ListResourceRecordSetsOutput, bool) bool, opts ...request.Option) {
		fn(returns, false)
	}

	r53.EXPECT().ListResourceRecordSetsPagesWithContext(ctx, gomock.Any(), gomock.Any()).Return(errTest).Do(do)
	de, err := d.GetResourceRecords(ctx, &route53.ListResourceRecordSetsInput{
		HostedZoneId: aws.String(d.ZoneID),
		MaxItems:     aws.String("300"), // 300 is max.
	}, true)
	assert.ErrorIs(err, errTest)
	assert.Nil(de)

	r53.EXPECT().ListResourceRecordSetsPagesWithContext(ctx, gomock.Any(), gomock.Any()).Return(nil).Do(do)
	de, err = d.GetResourceRecords(ctx, &route53.ListResourceRecordSetsInput{
		HostedZoneId: aws.String(d.ZoneID),
		MaxItems:     aws.String("300"), // 300 is max.
	}, true)
	assert.Nil(err)
	assert.Equal(*returns.ResourceRecordSets[0].Name, de[0].Subzone)
	assert.Equal(returns.ResourceRecordSets[0].TTL, de[0].TTL)
	assert.EqualValues(returns.ResourceRecordSets[0].ResourceRecords, de[0].Nameservers)
}

func TestNameServersString(t *testing.T) {
	t.Parallel()
	assert := assert.New(t)

	nss := delegate.NameServers{
		&route53.ResourceRecord{Value: aws.String("ns1")},
		&route53.ResourceRecord{Value: aws.String("ns2")},
		&route53.ResourceRecord{Value: aws.String("ns3")},
		&route53.ResourceRecord{Value: aws.String("ns4")},
	}

	assert.Equal("ns1, ns2, ns3, ns4", nss.String(), "string format is invalid")
}

func TestNameServersContains(t *testing.T) {
	t.Parallel()
	assert := assert.New(t)

	nss := delegate.NameServers{
		&route53.ResourceRecord{Value: aws.String("ns1.")},
		&route53.ResourceRecord{Value: aws.String("ns2.")},
		&route53.ResourceRecord{Value: aws.String("ns3.")},
		&route53.ResourceRecord{Value: aws.String("ns4.")},
	}

	assert.True(nss.Contains("ns1."))
	assert.True(nss.Contains("ns2."))
	assert.True(nss.Contains("ns3."))
	assert.True(nss.Contains("ns4."))
	assert.True(nss.Contains("ns3."))

	assert.False(nss.Contains("ns5."))
	assert.False(nss.Contains("ns8."))
	assert.False(nss.Contains("aws."))
	assert.False(nss.Contains("dns."))

	assert.False(nss.Contains("s."))
	assert.False(nss.Contains("ns."))
	assert.False(nss.Contains("ns"))
	assert.False(nss.Contains("s"))
	assert.False(nss.Contains(".."))
	assert.False(nss.Contains("."))
	assert.False(nss.Contains("*"))
	assert.False(nss.Contains(""))
}
