package delegate

import (
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/route53"
)

/* These procedures are used to delete a delegation. */

// Delete deletes a specific set of records from a specific recordset.
// If the records don't match it will throw an error.
func (d *Delegate) Delete(ctx aws.Context, request *Delegation) error {
	if request.TTL == nil {
		request.TTL = &d.TTL
	}

	_, err := d.Svc.ChangeResourceRecordSetsWithContext(ctx, &route53.ChangeResourceRecordSetsInput{
		HostedZoneId: aws.String(d.ZoneID),
		ChangeBatch: &route53.ChangeBatch{
			Changes: []*route53.Change{{
				Action: aws.String(route53.ChangeActionDelete),
				ResourceRecordSet: &route53.ResourceRecordSet{
					Name:            aws.String(request.Subzone),
					ResourceRecords: request.Nameservers,
					TTL:             request.TTL,
					Type:            aws.String(route53.RRTypeNs),
				},
			}},
			Comment: aws.String("AccountID: " + request.AccountID + ", ZoneID: " + request.ZoneID),
		},
	})
	if err != nil {
		return fmt.Errorf("ChangeResourceRecordSetsWithContext: %w", err)
	}

	return nil
}

// GetNameservers searches a zone for a record of type NS. Returns the nameservers.
// This is used immediately before the name servers are deleted.
func (d *Delegate) GetNameservers(ctx aws.Context, recordName string) (NameServers, error) {
	nss := NameServers{}

	err := d.Svc.ListResourceRecordSetsPagesWithContext(ctx, &route53.ListResourceRecordSetsInput{
		HostedZoneId: aws.String(d.ZoneID),
	}, func(rr *route53.ListResourceRecordSetsOutput, lastPage bool) bool {
		for _, recordset := range rr.ResourceRecordSets {
			if *recordset.Type == route53.RRTypeNs && *recordset.Name == recordName {
				nss = recordset.ResourceRecords

				return false
			}
		}

		return !lastPage
	})
	if err != nil {
		return nil, fmt.Errorf("ListResourceRecordSetsWithContext: %w", err)
	}

	if len(nss) == 0 {
		return nil, fmt.Errorf("%w: %s", ErrDelegationMissing, recordName)
	}

	return nss, nil
}
