package delegate_test

import (
	"testing"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/route53"
	gomock "github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
)

func TestDelete(t *testing.T) {
	t.Parallel()
	assert := assert.New(t)

	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	d, delegator := testNewConfig(mockCtrl)
	ctx := aws.BackgroundContext()
	req := &delegate.Delegation{
		ZoneID:    "JH32G4H3H234H34GH",
		Subzone:   "suuuuub.zoooon",
		AccountID: "312434324234",
		Nameservers: []*route53.ResourceRecord{
			{Value: aws.String("ns1.zone")},
			{Value: aws.String("ns2.zone")},
			{Value: aws.String("ns3.zone")},
		},
	}

	delegator.EXPECT().ChangeResourceRecordSetsWithContext(ctx, gomock.Any(), []request.Option{}).Do(
		func(_ aws.Context, obj *route53.ChangeResourceRecordSetsInput, _ ...request.Option) {
			assert.Equal(d.ZoneID, *obj.HostedZoneId)
			assert.Equal(route53.ChangeActionDelete, *obj.ChangeBatch.Changes[0].Action)
			assert.Equal(d.TTL, *obj.ChangeBatch.Changes[0].ResourceRecordSet.TTL)
			assert.Equal(route53.RRTypeNs, *obj.ChangeBatch.Changes[0].ResourceRecordSet.Type)
			assert.Equal(req.Subzone, *obj.ChangeBatch.Changes[0].ResourceRecordSet.Name)
		},
	).Return(nil, nil)
	assert.Nil(d.Delete(ctx, req), "this valid request must not produce an error")

	delegator.EXPECT().ChangeResourceRecordSetsWithContext(ctx, gomock.Any(), []request.Option{}).Return(nil, errTest)
	assert.ErrorIs(d.Delete(ctx, req), errTest, "the provided error must be returned.")
}

func TestGetNameservers(t *testing.T) {
	t.Parallel()
	assert := assert.New(t)

	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	d, delegator := testNewConfig(mockCtrl)
	ctx := aws.BackgroundContext()
	answers := []*string{
		aws.String("ns.server.1"),
		aws.String("ns.server.2"),
		aws.String("ns.server.3"),
		aws.String("ns.server.4"),
	}
	answer := &route53.ListResourceRecordSetsOutput{
		ResourceRecordSets: []*route53.ResourceRecordSet{{
			Type: aws.String(route53.RRTypeNs),
			Name: aws.String("subzoon.main.zone"),
			ResourceRecords: []*route53.ResourceRecord{
				{Value: answers[0]},
				{Value: answers[1]},
				{Value: answers[2]},
				{Value: answers[3]},
			},
		}},
	}
	input := &route53.ListResourceRecordSetsInput{HostedZoneId: aws.String(d.ZoneID)}
	do := func(ctx aws.Context, input *route53.ListResourceRecordSetsInput,
		fn func(*route53.ListResourceRecordSetsOutput, bool) bool, opts ...request.Option) {
		fn(answer, false)
	}

	delegator.EXPECT().ListResourceRecordSetsPagesWithContext(ctx, input, gomock.Any()).Return(errTest).Do(do)
	_, err := d.GetNameservers(ctx, "suuub.zoooon")
	assert.ErrorIs(err, errTest)

	delegator.EXPECT().ListResourceRecordSetsPagesWithContext(ctx, input, gomock.Any()).Return(nil).Do(do)
	_, err = d.GetNameservers(ctx, "suuub.zoooon.com")
	assert.ErrorIs(err, delegate.ErrDelegationMissing,
		"the returned answer does not contain the sub zone.")

	delegator.EXPECT().ListResourceRecordSetsPagesWithContext(ctx, input, gomock.Any()).Return(nil).Do(do)
	///
	answer.ResourceRecordSets[0].Name = aws.String("suuub.zoooon.com")
	_, err = d.GetNameservers(ctx, "suuub.zoooon.com")
	assert.Nil(err, "this valid request should not produce an error")
}
