package delegator

import (
	"errors"
	"fmt"
	"log"
	"strings"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"code.justin.tv/awsi/twitch-a2z-com/pkg/storage"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/route53"
)

// CreateDelegation does the following:
// - Grabs the zone and name servers from the remote account.
// - Verifies the delegation doesn't already exists and is a valid request.
// - Logs the request to S3.
// - Writes the Route53 RecordSet.
// - Logs the grant to S3.
// - Rolls back if logging to S3 fails.
func (c *Config) CreateDelegation(ctx aws.Context, req *Request) error {
	delegation, err := c.hw.GetRemoteZone(ctx, req)
	if err != nil {
		return fmt.Errorf("error checking zone: %w", err)
	}

	req.Subzone = delegation.Subzone // to be used as physical ID.

	err = c.hw.CheckNewZone(ctx, delegation)
	if err != nil {
		return fmt.Errorf("checking delegation failed: %w", err)
	}

	err = c.hw.Save(ctx, delegation, req.StackID, storage.Request)
	if err != nil {
		return fmt.Errorf("writing s3 request failed: %w", err)
	}

	log.Printf("Creating: %s (account: %s, id: %s)", req.Subzone, req.AccountID, req.SubZoneID)

	err = c.hw.Create(ctx, delegation)
	if err != nil {
		return fmt.Errorf("creating delegation failed: %w", err)
	}

	err = c.hw.Save(ctx, delegation, req.StackID, storage.Granted)
	if err != nil {
		// Rollback and report error.
		if rollbackErr := c.hw.Delete(ctx, delegation); rollbackErr != nil {
			s := rollbackErr.Error()

			return fmt.Errorf("writing s3 grant failed, rolled back: %w (rollback error: %v)", err, s)
		}

		return fmt.Errorf("writing s3 grant failed: %w (rolled back)", err)
	}

	return nil
}

// DeleteDelegation does the following:
// - Gets the name servers from a record in our main zone.
// - Gets the granted log from s3. This contains all delegation owner info.
// - Compares the account ID and zone ID from the granted log to the requestor's info.
// - If it matches, the delegation is deleted and the granted log is updated to "deleted".
func (c *Config) DeleteDelegation(ctx aws.Context, req *Request, newZoneID ...string) error {
	// Make sure the record from our zone matches the data provided from the requester.
	// The delegation must be granted (active) and the IDs must match.
	switch saved, err := c.hw.Get(ctx, req.AccountID, req.Subzone); {
	case err != nil:
		return fmt.Errorf("getting saved delegation: %w", err)
	case saved.Status == storage.Granted && len(newZoneID) > 0 && newZoneID[0] == saved.ZoneID:
		return ErrIgnored
	case saved.Status != storage.Granted:
		return fmt.Errorf("%w (status): %s != %s", ErrNotOwner, saved.Status, storage.Granted)
	case saved.ZoneID != req.SubZoneID:
		return fmt.Errorf("%w (zoneId): %s != %s", ErrNotOwner, saved.ZoneID, req.SubZoneID)
	case saved.StackID != req.StackID:
		return fmt.Errorf("%w (stackId): %s != %s", ErrNotOwner, saved.StackID, req.StackID)
	}

	nameservers, err := c.hw.GetNameservers(ctx, req.Subzone)
	if err != nil {
		return fmt.Errorf("getting name servers: %w", err)
	}

	delegation := &delegate.Delegation{
		AccountID:   req.AccountID,
		Nameservers: nameservers,
		Subzone:     req.Subzone,
		ZoneID:      req.SubZoneID,
	}

	log.Printf("Deleting: %s (account: %s, id: %s)", req.Subzone, req.AccountID, req.SubZoneID)

	err = c.hw.Delete(ctx, delegation)
	if err != nil {
		return fmt.Errorf("error deleting: %w", err)
	}

	// Ignore any s3 write error when deleting, but log it at least.
	err = c.hw.Save(ctx, delegation, req.StackID, storage.Deleted)
	if err != nil {
		log.Println("ERROR updating granted log to deleted:", err)
	}

	return nil
}

// UpdateDelegation checks if the update is a new zone name or not and behaves accordingly.
func (c *Config) UpdateDelegation(ctx aws.Context, req *Request, savedZoneID string) error {
	newZone, err := c.hw.GetRemoteZone(ctx, req)
	if err != nil {
		return fmt.Errorf("error checking zone: %w", err)
	}

	log.Printf("Updating: %s/%s => %s/%s", req.Subzone, savedZoneID, newZone.Subzone, req.SubZoneID)

	if oldSubZone := req.Subzone; newZone.Subzone != oldSubZone {
		// If the two zone names are different the cfn custom resource behaves differently.
		// Do not delete the old one here. The lambda will trigger again with a delete operation.
		err := c.hw.CreateDelegation(ctx, req)
		if err != nil {
			// Do not change the physical resource ID if creating new zone fails.
			req.Subzone = oldSubZone

			return fmt.Errorf("creating delegation: %w", err)
		}

		return nil
	}

	// At this point it's a new zone ID but the same zone name, so we need to clear it out.
	savedZoneID, req.SubZoneID = req.SubZoneID, savedZoneID // swap

	err = c.hw.DeleteDelegation(ctx, req, savedZoneID)
	if errors.Is(err, ErrIgnored) {
		log.Printf("Ignoring: (EXISTS) %s (account: %s, id: %s)", req.Subzone, req.AccountID, savedZoneID)
		// This only happens during a rollback request.
		// We don't have to roll anything back, so we ignore it and finish up.
		return nil
	} else if err != nil {
		return fmt.Errorf("deleting delegation: %w", err)
	}

	req.SubZoneID = savedZoneID // swap it back.

	return c.hw.CreateDelegation(ctx, req) // create it.
}

// GetRemoteZone determines if the payload contains the remogte zone info or not.
// If it's not included, it's fetched by assuming an IAM role in the requesting account.
// If it included, it's formatted and returned.
func (c *Config) GetRemoteZone(ctx aws.Context, req *Request) (*delegate.Delegation, error) {
	// Some requests come with a zone name and servers (so we have everything up front).
	// These requests were not validated with an IAM Role. That feature is off by default.
	if req.Newzone != "" && len(req.NSs) > 3 {
		return &delegate.Delegation{
			AccountID:   req.AccountID,
			Subzone:     strings.TrimSuffix(req.Newzone, ".") + ".", // must end with a dot.
			ZoneID:      req.SubZoneID,
			Nameservers: req.NSs,
		}, nil
	}

	var (
		arn = fmt.Sprintf("arn:aws:iam::%s:role/%s%s", req.AccountID, c.RolePrefix, req.SubZoneID)
		ses = session.Must(session.NewSession())
		svc = route53.New(ses,
			aws.NewConfig().WithCredentials(stscreds.NewCredentials(ses, arn)).WithRegion(req.Region).WithMaxRetries(Retries))
	)

	newZone, err := c.hw.GetZone(ctx, req.AccountID, req.SubZoneID, svc)
	if err != nil {
		req.Subzone = req.SubZoneID // Set this so we return _something_ for a PhysicalResourceID.

		return nil, fmt.Errorf("getting subzone info: %s: %w", req.SubZoneID, err)
	}

	return newZone, nil
}
