package delegator_test

import (
	"testing"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegator"
	"code.justin.tv/awsi/twitch-a2z-com/pkg/storage"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/route53"
	gomock "github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
)

func testReq() (*delegator.Request, *delegate.Delegation) {
	req := &delegator.Request{
		StackID:   "aws:arn:cfn:us-west-2:8371261278122:stuff",
		AccountID: "8371261278122",
		SubZoneID: "50M34007Z00NLD",
		Subzone:   "sub.main.zone.",
		Region:    "us-west-2",
		Newzone:   "passed.in.without.iam.role.",
		NSs: delegate.NameServers{
			&route53.ResourceRecord{Value: aws.String("name.server.1")},
			&route53.ResourceRecord{Value: aws.String("name.server.2")},
			&route53.ResourceRecord{Value: aws.String("name.server.3")},
			&route53.ResourceRecord{Value: aws.String("name.server.4")},
		},
	}

	return req, &delegate.Delegation{
		AccountID:   req.AccountID,
		Subzone:     req.Newzone,
		ZoneID:      req.SubZoneID,
		Nameservers: req.NSs,
	}
}

// testSetup provides the standard data bits to run tests.
func testSetup(t *testing.T) (*assert.Assertions, *gomock.Controller, *delegator.Request, *delegate.Delegation, aws.Context) { //nolint: lll
	t.Helper()

	req, delegation := testReq()

	return assert.New(t), gomock.NewController(t), req, delegation, aws.BackgroundContext()
}

func TestCreateDelegation(t *testing.T) {
	t.Parallel()

	assert, mockCtrl, req, delegation, ctx := testSetup(t)
	defer mockCtrl.Finish()

	c, mockdelegator := testNewConfig(mockCtrl)

	gomock.InOrder(
		// This is the expected current order of operations when creating a delegation.
		mockdelegator.EXPECT().GetRemoteZone(ctx, req).Return(delegation, nil),     // get remote zone name/NSs
		mockdelegator.EXPECT().CheckNewZone(ctx, delegation),                       // make sure it's valid
		mockdelegator.EXPECT().Save(ctx, delegation, req.StackID, storage.Request), // save 'request' in s3
		mockdelegator.EXPECT().Create(ctx, delegation),                             // create delegation
		mockdelegator.EXPECT().Save(ctx, delegation, req.StackID, storage.Granted). // update s3: 'granted'
												Return(errTest), // return an error here to trigger rollback.
		mockdelegator.EXPECT().Delete(ctx, delegation), // rollback due to error
	)
	assert.ErrorIs(c.CreateDelegation(ctx, req), errTest, "the error we passed in should be returned.")

	mockdelegator.EXPECT().GetRemoteZone(ctx, req).Return(nil, errTest)
	assert.ErrorIs(c.CreateDelegation(ctx, req), errTest, "the error we passed in should be returned.")

	gomock.InOrder(
		mockdelegator.EXPECT().GetRemoteZone(ctx, req).Return(delegation, nil), // get remote zone name/NSs
		mockdelegator.EXPECT().CheckNewZone(ctx, delegation).Return(errTest),   // ERROR: make sure it's valid
	)
	assert.ErrorIs(c.CreateDelegation(ctx, req), errTest, "the error we passed in should be returned.")

	gomock.InOrder(
		mockdelegator.EXPECT().GetRemoteZone(ctx, req).Return(delegation, nil), // get remote zone name/NSs
		mockdelegator.EXPECT().CheckNewZone(ctx, delegation),                   // make sure it's valid
		mockdelegator.EXPECT().Save(ctx, delegation, req.StackID, storage.Request). // ERROR: save 'request' in s3
												Return(errTest),
	)
	assert.ErrorIs(c.CreateDelegation(ctx, req), errTest, "the error we passed in should be returned.")

	gomock.InOrder(
		mockdelegator.EXPECT().GetRemoteZone(ctx, req).Return(delegation, nil),     // get remote zone name/NSs
		mockdelegator.EXPECT().CheckNewZone(ctx, delegation),                       // make sure it's valid
		mockdelegator.EXPECT().Save(ctx, delegation, req.StackID, storage.Request), // save 'request' in s3
		mockdelegator.EXPECT().Create(ctx, delegation).Return(errTest),             // ERROR: create delegation
	)
	assert.ErrorIs(c.CreateDelegation(ctx, req), errTest, "the error we passed in should be returned.")

	// This last request is 100% valid, no errors are forced to return.
	gomock.InOrder(
		mockdelegator.EXPECT().GetRemoteZone(ctx, req).Return(delegation, nil),     // get remote zone name/NSs
		mockdelegator.EXPECT().CheckNewZone(ctx, delegation),                       // make sure it's valid
		mockdelegator.EXPECT().Save(ctx, delegation, req.StackID, storage.Request), // save 'request' in s3
		mockdelegator.EXPECT().Create(ctx, delegation),                             // create delegation
		mockdelegator.EXPECT().Save(ctx, delegation, req.StackID, storage.Granted), // update s3: 'granted'
	)
	assert.Nil(c.CreateDelegation(ctx, req), "this valid request must not produce an error")
}

func TestDeleteDelegation(t *testing.T) { // nolint: funlen
	t.Parallel()

	assert, mockCtrl, req, _, ctx := testSetup(t)
	defer mockCtrl.Finish()

	c, mockdelegator := testNewConfig(mockCtrl)
	saved, delegation := &storage.StorePayload{
		ZoneID:  req.SubZoneID,
		StackID: req.StackID,
		Status:  storage.Granted,
	}, &delegate.Delegation{
		AccountID:   req.AccountID,
		Nameservers: nil,
		Subzone:     req.Subzone,
		ZoneID:      req.SubZoneID,
	}

	gomock.InOrder( // This first test is a valid delete request. No forced errors returned.
		mockdelegator.EXPECT().Get(ctx, req.AccountID, req.Subzone).Return(saved, nil),
		mockdelegator.EXPECT().GetNameservers(ctx, req.Subzone).Return(nil, nil),
		mockdelegator.EXPECT().Delete(ctx, delegation).Return(nil),
		mockdelegator.EXPECT().Save(ctx, delegation, req.StackID, storage.Deleted).Return(nil),
	)
	assert.Nil(c.DeleteDelegation(ctx, req))

	// Test Various Low-Level errors.
	gomock.InOrder( // Force error on the Save method.
		mockdelegator.EXPECT().Get(ctx, req.AccountID, req.Subzone).Return(saved, nil),
		mockdelegator.EXPECT().GetNameservers(ctx, req.Subzone).Return(nil, nil),
		mockdelegator.EXPECT().Delete(ctx, delegation).Return(nil),
		mockdelegator.EXPECT().Save(ctx, delegation, req.StackID, storage.Deleted).Return(errTest),
	)
	assert.Nil(c.DeleteDelegation(ctx, req), "a storage update error should not be returned")

	gomock.InOrder( // Force error on the Delete method.
		mockdelegator.EXPECT().Get(ctx, req.AccountID, req.Subzone).Return(saved, nil),
		mockdelegator.EXPECT().GetNameservers(ctx, req.Subzone).Return(nil, nil),
		mockdelegator.EXPECT().Delete(ctx, delegation).Return(errTest),
	)
	assert.ErrorIs(c.DeleteDelegation(ctx, req), errTest, "the error we passed in should be returned.")

	gomock.InOrder( // Force error on the GetNameservers method.
		mockdelegator.EXPECT().Get(ctx, req.AccountID, req.Subzone).Return(saved, nil),
		mockdelegator.EXPECT().GetNameservers(ctx, req.Subzone).Return(nil, errTest),
	)
	assert.ErrorIs(c.DeleteDelegation(ctx, req), errTest, "the error we passed in should be returned.")

	// Force error on the Get method.
	mockdelegator.EXPECT().Get(ctx, req.AccountID, req.Subzone).Return(nil, errTest)
	assert.ErrorIs(c.DeleteDelegation(ctx, req), errTest, "the error we passed in should be returned.")

	// Test for an ignorable situation.
	mockdelegator.EXPECT().Get(ctx, req.AccountID, req.Subzone).Return(saved, nil)
	assert.ErrorIs(c.DeleteDelegation(ctx, req, req.SubZoneID), delegator.ErrIgnored,
		"if the new zone ID (during an update) matches the existing delegation, ignore it and do not delete!")

	/* Test Grant Ownership Errors */
	saved.Status = "Invalid" // an invalid status invalidates ownership
	mockdelegator.EXPECT().Get(ctx, req.AccountID, req.Subzone).Return(saved, nil)
	assert.ErrorIs(c.DeleteDelegation(ctx, req), delegator.ErrNotOwner,
		"an invalid status must produce a not-owner error")

	saved.Status, saved.ZoneID = storage.Granted, saved.Status // an invalid zone ID invalidates ownership
	mockdelegator.EXPECT().Get(ctx, req.AccountID, req.Subzone).Return(saved, nil)
	assert.ErrorIs(c.DeleteDelegation(ctx, req), delegator.ErrNotOwner,
		"an invalid zone ID must produce a not-owner error")

	saved.ZoneID, saved.StackID = req.SubZoneID, saved.ZoneID // an invalid stack ID invalidates ownership
	mockdelegator.EXPECT().Get(ctx, req.AccountID, req.Subzone).Return(saved, nil)
	assert.ErrorIs(c.DeleteDelegation(ctx, req), delegator.ErrNotOwner,
		"an invalid stack ID must produce a not-owner error")
}

func TestUpdateDelegation(t *testing.T) {
	t.Parallel()

	assert, mockCtrl, req, newZone, ctx := testSetup(t)
	defer mockCtrl.Finish()

	c, mockdelegator := testNewConfig(mockCtrl)

	newZone.Subzone = req.Subzone // make them match for the first run
	gomock.InOrder(               // This first test is a valid delete request. No forced errors returned.
		mockdelegator.EXPECT().GetRemoteZone(ctx, req).Return(newZone, nil),          // get remote zone name/NSs
		mockdelegator.EXPECT().DeleteDelegation(ctx, req, req.SubZoneID).Return(nil), // Delete old delegation.
		mockdelegator.EXPECT().CreateDelegation(ctx, req).Return(nil),                // Create new delegation.
	)
	assert.Nil(c.UpdateDelegation(ctx, req, req.SubZoneID))

	newZone.Subzone = req.Newzone
	gomock.InOrder( // This first test is a valid delete request. No forced errors returned.
		mockdelegator.EXPECT().GetRemoteZone(ctx, req).Return(newZone, nil), // get remote zone name/NSs
		// We do not delete the old delegation when the names are different.
		mockdelegator.EXPECT().CreateDelegation(ctx, req).Return(nil), // Create new delegation.
	)
	assert.Nil(c.UpdateDelegation(ctx, req, req.SubZoneID))

	// Make sure ignorable conditions do not return an error.
	newZone.Subzone = req.Subzone
	gomock.InOrder(
		mockdelegator.EXPECT().GetRemoteZone(ctx, req).Return(newZone, nil),
		mockdelegator.EXPECT().DeleteDelegation(ctx, req, req.SubZoneID).Return(delegator.ErrIgnored),
	)
	assert.Nil(c.UpdateDelegation(ctx, req, req.SubZoneID), "ignored delete errors should not trigger a create event")

	/* Now test a few error conditions. */

	newZone.Subzone = req.Subzone
	mockdelegator.EXPECT().GetRemoteZone(ctx, req).Return(nil, errTest)
	assert.ErrorIs(c.UpdateDelegation(ctx, req, req.SubZoneID), errTest, "the error we passed in should be returned.")

	gomock.InOrder(
		mockdelegator.EXPECT().GetRemoteZone(ctx, req).Return(newZone, nil),
		mockdelegator.EXPECT().DeleteDelegation(ctx, req, req.SubZoneID).Return(errTest),
	)
	assert.ErrorIs(c.UpdateDelegation(ctx, req, req.SubZoneID), errTest, "the error we passed in should be returned.")

	newZone.Subzone = req.Newzone + "different.names."
	before := req.Subzone
	gomock.InOrder(
		mockdelegator.EXPECT().GetRemoteZone(ctx, req).Return(newZone, nil), // get remote zone name/NSs
		mockdelegator.EXPECT().CreateDelegation(ctx, req).Return(errTest),   // Create new delegation.
	)
	assert.ErrorIs(c.UpdateDelegation(ctx, req, req.SubZoneID), errTest)
	assert.Equal(before, req.Subzone, "the zone name must not be changed on failed create")
}

// I cannot find a way to dig the fabiracated ARN out of this method.
func TestGetRemoteZone(t *testing.T) {
	t.Parallel()

	assert, mockCtrl, req, newZone, ctx := testSetup(t)
	defer mockCtrl.Finish()

	// In this test, the remote zone data is in the `req` request.
	c, mockdelegator := testNewConfig(mockCtrl)
	newZoneReturned, err := c.GetRemoteZone(ctx, req)
	assert.Nil(err, "there must not be an error returned")
	assert.Equal(newZone, newZoneReturned, "an invalid zone or zone data was returned")

	// In this test, the remote zone data is NOT in the `req` request, fetch it.
	req.Newzone = ""
	mockdelegator.EXPECT().GetZone(ctx, req.AccountID, req.SubZoneID, gomock.Any()).Return(newZone, nil)
	newZoneReturned, err = c.GetRemoteZone(ctx, req)
	assert.Nil(err, "there must not be an error returned")
	assert.Equal(newZone, newZoneReturned, "an invalid zone or zone data was returned")

	// In this test, fetching the zone produces an error that must be returned.
	mockdelegator.EXPECT().GetZone(ctx, req.AccountID, req.SubZoneID, gomock.Any()).Return(nil, errTest)
	newZoneReturned, err = c.GetRemoteZone(ctx, req)
	assert.Nil(newZoneReturned, "returned data must be nil when there's an error")
	assert.ErrorIs(err, errTest, "the error we passed in should be returned.")
}
