package delegator

import (
	"fmt"
	"log"
	"os"
	"strings"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"code.justin.tv/awsi/twitch-a2z-com/pkg/metrics"
	"code.justin.tv/awsi/twitch-a2z-com/pkg/storage"
	"github.com/aws/aws-lambda-go/cfn"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/aws/aws-sdk-go/service/route53"
	"github.com/aws/aws-sdk-go/service/s3"
)

// Config makes this module go brrrr.
// Careful what goes in here. This may be re-used for _different_ requests.
type Config struct {
	RolePrefix     string
	AllowNoIAMRole bool // setting this to true may cause security incidents.
	Delegate       *delegate.Delegate
	Storage        *storage.Storage
	Metrics        *metrics.Metrics
	hw             Handler
}

// Request contains the data to uniquely identify a request.
type Request struct {
	StackID   string
	AccountID string
	SubZoneID string
	Subzone   string
	Region    string
	Newzone   string
	NSs       delegate.NameServers
}

// Errors.
var (
	ErrNoAccountID        = fmt.Errorf("no account ID found in payload")
	ErrNoZoneID           = fmt.Errorf("no zone ID found in payload")
	ErrUnknownRequestType = fmt.Errorf("unknown request type")
	ErrNotOwner           = fmt.Errorf("account requesting update is not the account that made the delegation")
	ErrIgnored            = fmt.Errorf("this request has been ignored") // rollback requests can be ignored.
)

const (
	// Retries for aws session calls.
	Retries = 5
	// RegionLoc is used to find the region in the Stack ID.
	RegionLoc = 3
	// AccountLoc is used to find the Account ID in the Stack ID.
	AccountLoc = 4
	// MetricNamespace is for CloudWatch Metrics.
	MetricNamespace = "Twitch/Delegator"
)

// New fills out all three library config structs for the Delegator Lambda.
func New(sess client.ConfigProvider, awsc *aws.Config) *Config {
	return &Config{
		RolePrefix:     os.Getenv("ROLEPFX"), // used if request does not contain zone name or name servers.
		AllowNoIAMRole: os.Getenv("ALLOW_NO_IAM_ROLE") == "true",
		Delegate: &delegate.Delegate{
			Svc:      route53.New(sess, awsc),
			ZoneName: os.Getenv("ZONENAME"), // optional, looked up from zoneID if missing.
			ZoneID:   os.Getenv("ZONEID"),   // not optional.
			TTL:      delegate.DefaultTTL,
		},
		Storage: &storage.Storage{
			Svc:    s3.New(sess, awsc),
			Prefix: os.Getenv("S3_KEYPREFIX"), // used as a path prefix in the s3 bucket.
			Bucket: os.Getenv("S3_BUCKET"),    // not optional.
		},
		Metrics: &metrics.Metrics{
			Svc:       cloudwatch.New(sess, awsc),
			ZoneID:    os.Getenv("ZONEID"), // not optional, same as above.
			Namespace: MetricNamespace,
		},
	}
}

// LambdaHandler checks the payload and sets up the delegation.
// https://godoc.org/github.com/aws/aws-lambda-go/cfn
// == You must always return a physical resource ID, even if there is an error.
// == Otherwise, the error does not make it back to the caller.
func (c *Config) LambdaHandler(ctx aws.Context, data cfn.Event) (string, map[string]interface{}, error) {
	c.SetHandler()                                    // set the interface.
	c.hw.Send(string(data.RequestType), "all", "all") // write a request metric

	switch data.RequestType { // Write a log about this incoming request.
	case cfn.RequestCreate:
		log.Printf("Create Request: ZoneID: %s, StackID: %s\n", data.ResourceProperties["ZoneID"], data.StackID)
	case cfn.RequestDelete:
		log.Printf("Delete Request: ZoneID: %s, StackID: %s, ZoneName: %s\n",
			data.ResourceProperties["ZoneID"], data.StackID, data.PhysicalResourceID)
	case cfn.RequestUpdate:
		log.Printf("Update Request: ZoneID: %s, StackID: %s, ZoneName: %s, new ZoneID: %s\n",
			data.OldResourceProperties["ZoneID"], data.StackID,
			data.PhysicalResourceID, data.ResourceProperties["ZoneID"])
	default:
		log.Printf("Unknown Request: %s, StackID: %s, ResourceProperties: %v\n",
			data.ResourceType, data.StackID, data.ResourceProperties)
	}

	err := c.hw.SaveOwnZone(ctx)

	defer func() {
		if err != nil {
			c.hw.Send("ERROR", "all", "all")
		}
	}()

	if err != nil {
		return data.StackID, nil, fmt.Errorf("unable to get own zone data for %s: %w", c.Delegate.ZoneID, err)
	}

	stackSplit := strings.Split(data.StackID, ":")
	if len(stackSplit) < 5 { // nolint: gomnd
		// This is rather unlikely, and can only happen on a forged request.
		return data.StackID, nil, ErrNoAccountID
	}

	newzone, nameservers := c.GetZoneDataFromReq(data)
	request := &Request{
		AccountID: stackSplit[AccountLoc],  // This becomes the zone owner1.
		Region:    stackSplit[RegionLoc],   // For assume role, but probably useless.
		StackID:   data.StackID,            // This becomes the zone owner2.
		Subzone:   data.PhysicalResourceID, // Only exists on update & delete.
		Newzone:   newzone,                 // Only added if allowing no IAM role.
		NSs:       nameservers,             // Only added if allowing no IAM role.
	}

	physicalResourceID, err := c.finishRequest(ctx, data, request)
	if err == nil { // NOT nil.
		c.hw.Send(string(data.RequestType), request.Subzone, request.AccountID)
	}

	return physicalResourceID, nil, err
}

func (c *Config) finishRequest(ctx aws.Context, data cfn.Event, req *Request) (string, error) {
	if req.SubZoneID, _ = data.ResourceProperties["ZoneID"].(string); req.SubZoneID == "" {
		return "unknown zone in account " + req.AccountID, ErrNoZoneID
	}

	// Handle CRUD.
	switch data.RequestType {
	default:
		return "zone ID " + req.SubZoneID, fmt.Errorf("%w: %s", ErrUnknownRequestType, data.RequestType)
	case cfn.RequestCreate:
		// CreateDelegation updates req.Subzone.
		return req.Subzone, c.hw.CreateDelegation(ctx, req)
	case cfn.RequestDelete:
		return req.Subzone, c.hw.DeleteDelegation(ctx, req)
	case cfn.RequestUpdate:
		// Pass in the old zone id on an update. If it's missing (invalid request) this app will crash.
		return req.Subzone, c.hw.UpdateDelegation(ctx, req, data.OldResourceProperties["ZoneID"].(string))
	}
}

// GetZoneDataFromReq returns zone data from the request if it exists and is allowed.
// Normally we use an IAM role to get zone data.
func (c *Config) GetZoneDataFromReq(data cfn.Event) (string, delegate.NameServers) {
	if !c.AllowNoIAMRole {
		return "", nil
	}

	// Newzone and Nameservers can be used to setup delegations without validating the zone using an IAM role.
	// Potential security risk.
	var (
		newzone, _     = data.ResourceProperties["ZoneName"].(string)
		nameservers, _ = data.ResourceProperties["NameServers"].([]interface{})
		nss            = delegate.NameServers{}
	)

	for i := range nameservers {
		if ns, _ := nameservers[i].(string); ns != "" {
			nss = append(nss, &route53.ResourceRecord{Value: &ns})
		}
	}

	return newzone, nss
}
