package delegator_test

import (
	"fmt"
	"os"
	"strconv"
	"strings"
	"testing"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegator"
	"code.justin.tv/awsi/twitch-a2z-com/pkg/mocks"
	"code.justin.tv/awsi/twitch-a2z-com/pkg/storage"
	"github.com/aws/aws-lambda-go/cfn"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/route53"
	gomock "github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
)

var errTest = fmt.Errorf("this is a test error")

// This is used by all the tests to create mock interfaces.
func testNewConfig(mockCtrl *gomock.Controller) (*delegator.Config, *mocks.MockHandler) {
	hw := mocks.NewMockHandler(mockCtrl)
	config := &delegator.Config{
		AllowNoIAMRole: true, // good for testing, not for prod.
		RolePrefix:     "iam-role-prefix-",
		Delegate: &delegate.Delegate{
			ZoneName: "main.zone.",
			ZoneID:   "M41N-V3RYC007Z0N31D",
		},
		Storage: &storage.Storage{
			Bucket: "some-bucket",
			Prefix: "delegations",
		},
	}

	config.SetHandler(hw) // provide mocks as an interface delegator.

	return config, hw
}

//nolint: paralleltest
func TestNewDelegator(t *testing.T) {
	var (
		assert = assert.New(t)
		sess   = session.Must(session.NewSession())
		awsc   = aws.NewConfig().WithMaxRetries(delegator.Retries)
	)

	os.Setenv("ROLEPFX", "ROLEPFX")
	os.Setenv("ALLOW_NO_IAM_ROLE", "false")
	os.Setenv("ZONENAME", "ZONENAME")
	os.Setenv("ZONEID", "ZONEID")
	os.Setenv("S3_KEYPREFIX", "S3_KEYPREFIX")
	os.Setenv("S3_BUCKET", "S3_BUCKET")

	c := delegator.New(sess, awsc)

	assert.Equal(os.Getenv("ROLEPFX"), c.RolePrefix)
	assert.Equal(os.Getenv("ALLOW_NO_IAM_ROLE"), strconv.FormatBool(c.AllowNoIAMRole))
	assert.Equal(os.Getenv("ZONENAME"), c.Delegate.ZoneName)
	assert.Equal(os.Getenv("ZONEID"), c.Delegate.ZoneID)
	assert.Equal(os.Getenv("S3_KEYPREFIX"), c.Storage.Prefix)
	assert.Equal(os.Getenv("S3_BUCKET"), c.Storage.Bucket)
	assert.Equal(delegate.DefaultTTL, c.Delegate.TTL)
	assert.NotNil(c.Delegate.Svc)
	assert.NotNil(c.Metrics.Svc)
	assert.NotNil(c.Storage.Svc)
}

// This test was written before the method was broken up into more methods.
// This is kinda big, but tests basically everything in delegator.go.
func TestLambdaHandler(t *testing.T) { //nolint: funlen
	t.Parallel()
	a := assert.New(t)

	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	c, mockdelegator := testNewConfig(mockCtrl)
	ctx := aws.BackgroundContext()
	// Create fake input data. This mimics what comes from the SNS topic.
	data := cfn.Event{
		RequestType:        cfn.RequestUpdate,
		RequestID:          "unused",
		ResponseURL:        "https://something.something",
		ResourceType:       "Things",
		PhysicalResourceID: "old.subzone.goes.here", // this is normally empty on a create request.
		LogicalResourceID:  "unused",
		StackID:            "arn:aws:cloudformation:region:ACCOUNT_ID:stack/STACK_NAME/STACK_UUID",
		ResourceProperties: map[string]interface{}{
			"ZoneID":      "SU8Z0N31D",
			"ZoneName":    "my.zone.name",
			"NameServers": []interface{}{"name.server.1", "name.server.2", "name.server.3", "name.server.4"},
		},
		OldResourceProperties: map[string]interface{}{
			"ZoneID": "07DZ0N31DH3R3", // this is normally empty on a create request.
		},
	}

	// This is the data we expect to be passed into the mocked methods.
	// Copied from the data we pass into the primary methods.
	expected := &delegator.Request{
		StackID:   data.StackID,
		AccountID: strings.Split(data.StackID, ":")[delegator.AccountLoc],
		SubZoneID: data.ResourceProperties["ZoneID"].(string),
		Subzone:   data.PhysicalResourceID,
		Region:    strings.Split(data.StackID, ":")[delegator.RegionLoc],
		Newzone:   data.ResourceProperties["ZoneName"].(string),
		NSs: delegate.NameServers{
			&route53.ResourceRecord{Value: aws.String(data.ResourceProperties["NameServers"].([]interface{})[0].(string))},
			&route53.ResourceRecord{Value: aws.String(data.ResourceProperties["NameServers"].([]interface{})[1].(string))},
			&route53.ResourceRecord{Value: aws.String(data.ResourceProperties["NameServers"].([]interface{})[2].(string))},
			&route53.ResourceRecord{Value: aws.String(data.ResourceProperties["NameServers"].([]interface{})[3].(string))},
		},
	}
	// Make sure the data we passed into this method is passed into the next method correctly.
	gomock.InOrder(
		mockdelegator.EXPECT().Send(string(cfn.RequestUpdate), "all", "all"),
		mockdelegator.EXPECT().SaveOwnZone(ctx),
		mockdelegator.EXPECT().UpdateDelegation(ctx, expected, data.OldResourceProperties["ZoneID"].(string)).Times(1),
		mockdelegator.EXPECT().Send(string(cfn.RequestUpdate), expected.Subzone, expected.AccountID),
	)

	id, m, err := c.LambdaHandler(ctx, data)
	a.Nil(m, "The return-values map must be empty. This app does not return values.")
	a.Nil(err, "There must not be an error. This was a mostly valid request.")
	a.EqualValues(data.PhysicalResourceID, id)

	// Make sure we get create and delete paths!
	data.RequestType = cfn.RequestCreate
	//
	gomock.InOrder(
		mockdelegator.EXPECT().Send(string(data.RequestType), "all", "all"),
		mockdelegator.EXPECT().SaveOwnZone(ctx),
		mockdelegator.EXPECT().CreateDelegation(ctx, expected).Times(1),
		mockdelegator.EXPECT().Send(string(data.RequestType), expected.Subzone, expected.AccountID),
	)

	id, _, err = c.LambdaHandler(ctx, data)
	a.Nil(err, "There must not be an error. This was a mostly valid request.")
	a.EqualValues(expected.Subzone, id, "the subzone must be returned as the physical ID.")

	// Change req type to delete.
	data.RequestType = cfn.RequestDelete
	//
	gomock.InOrder(
		mockdelegator.EXPECT().Send(string(data.RequestType), "all", "all"),
		mockdelegator.EXPECT().SaveOwnZone(ctx),
		mockdelegator.EXPECT().DeleteDelegation(ctx, expected).Times(1),
		mockdelegator.EXPECT().Send(string(data.RequestType), expected.Subzone, expected.AccountID),
	)

	id, _, err = c.LambdaHandler(ctx, data)
	a.Nil(err, "There must not be an error. This was a mostly valid request.")
	a.EqualValues(expected.Subzone, id, "the subzone must be returned as the physical ID.")

	// Make sure we return an error with c.Delegate.Setup/GetZone fails.
	gomock.InOrder(
		mockdelegator.EXPECT().Send(string(data.RequestType), "all", "all"),
		mockdelegator.EXPECT().SaveOwnZone(ctx).Return(errTest),
		mockdelegator.EXPECT().Send("ERROR", "all", "all"),
	)
	//
	id, _, err = c.LambdaHandler(ctx, data)
	a.ErrorIs(err, errTest, "The error passed in must be returned.")
	a.EqualValues(expected.StackID, id, "the stack ID must be returned as the physical ID when the zone name lookup fails")

	/* Test the errors this method produces. */
	data.RequestType = "Invalid Request"
	mockdelegator.EXPECT().Send(string(data.RequestType), "all", "all").AnyTimes()
	mockdelegator.EXPECT().SaveOwnZone(ctx).AnyTimes()
	mockdelegator.EXPECT().Send("ERROR", "all", "all").AnyTimes()

	id, _, err = c.LambdaHandler(ctx, data)
	a.ErrorIs(err, delegator.ErrUnknownRequestType)
	a.EqualValues("zone ID SU8Z0N31D", id)

	data.StackID = "invalid"
	id, _, err = c.LambdaHandler(ctx, data)
	a.ErrorIs(err, delegator.ErrNoAccountID)
	a.EqualValues("invalid", id,
		"The physical resource ID must be set to 'invalid' when the account ID and zone name are both unknown.")

	data.StackID = "arn:aws:cloudformation:region:ACCOUNT_ID:"
	delete(data.ResourceProperties, "ZoneID")
	id, _, err = c.LambdaHandler(ctx, data)
	a.ErrorIs(err, delegator.ErrNoZoneID)
	a.EqualValues("unknown zone in account ACCOUNT_ID", id,
		"The physical resource ID must be set to useful identfiable string when the zone name is not yet known.")
}

func TestTetZoneDataFromReq(t *testing.T) {
	t.Parallel()
	assert := assert.New(t)

	c := &delegator.Config{AllowNoIAMRole: false}
	s, nss := c.GetZoneDataFromReq(cfn.Event{
		ResourceProperties: map[string]interface{}{
			"ZoneName":    "zone.name",
			"NameServers": []interface{}{"ns1", "ns2"},
		},
	})

	assert.Nil(nss, "name servers must always be empty when AllowNoIAMRole is false")
	assert.Empty(s, "zone name must always be empty when AllowNoIAMRole is false")
}
