// Package delidangle provides the scanner and worker lambdas that scan, check for,
// and delete dangling delegations in public hosted zones.
package delidangle

//nolint:lll
//go:generate mockgen -destination=../mocks/mock_delidangle.go -package=mocks code.justin.tv/awsi/twitch-a2z-com/pkg/delidangle Resolver,Queuer,Notifier

import (
	"context"
	"os"
	"strconv"
	"time"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/route53"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/miekg/dns"
)

// DNSPort is 53.
const DNSPort = ":53" // added to server names to connect to them.

// Config is the data needed to make this package go brrr.
type Config struct {
	SafeMode bool
	SQSToken string
	SNSToken string
	SNS      Notifier
	SQS      Queuer
	DNS      Resolver
	R53      *delegate.Delegate
}

// New creates a Dangling Delegation Checker config.
func New(sess client.ConfigProvider, awsc *aws.Config) *Config {
	safe, err := strconv.ParseBool(os.Getenv("SAFEMODE"))
	if err != nil {
		safe = true
	}

	return &Config{
		SafeMode: safe,
		SNSToken: os.Getenv("SNS_TOKEN"),
		SQSToken: os.Getenv("SQS_TOKEN"),
		SQS:      sqs.New(sess, awsc),
		DNS:      &dns.Client{Net: "udp"}, // nolint: exhaustivestruct
		SNS:      sns.New(sess, awsc),
		R53: &delegate.Delegate{
			Svc:      route53.New(sess, awsc),
			ZoneName: os.Getenv("ZONENAME"), // optional, looked up from zoneID if missing.
			ZoneID:   os.Getenv("ZONEID"),   // not optional.
			TTL:      delegate.DefaultTTL,
		},
	}
}

// Resolver allows mocking DNS lookups for tests.
type Resolver interface {
	ExchangeContext(ctx context.Context, m *dns.Msg, a string) (r *dns.Msg, rtt time.Duration, err error)
}

// Queuer allows mocking our AWS SQS calls for tests.
type Queuer interface {
	SendMessageWithContext(ctx context.Context, input *sqs.SendMessageInput,
		_ ...request.Option) (*sqs.SendMessageOutput, error)
}

// Notifier allows mocking our calls to SNS.
type Notifier interface {
	Publish(input *sns.PublishInput) (*sns.PublishOutput, error)
}
