package delidangle

import (
	"context"
	"encoding/json"
	"fmt"
	"log"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/route53"
	"github.com/aws/aws-sdk-go/service/sqs"
)

// ScannerHandler is the lambda handler for the scanner app.
func (c *Config) ScannerHandler(ctx context.Context) error {
	if err := c.R53.SaveOwnZone(ctx); err != nil {
		return fmt.Errorf("unable to get own zone data for %s: %w", c.R53.ZoneID, err)
	}

	delegations, err := c.R53.GetResourceRecords(ctx,
		&route53.ListResourceRecordSetsInput{HostedZoneId: &c.R53.ZoneID}, false)
	if err != nil {
		return fmt.Errorf("error scanning zone %s: %w", c.R53.ZoneName, err)
	}

	ok, bad := c.WriteSQS(ctx, delegations)
	log.Printf("Sent %d/%d Delegations to SQS; %d failed.", ok, len(delegations), bad)

	return nil
}

// WriteSQS sends a delegation payload to an SQS queue.
func (c *Config) WriteSQS(ctx context.Context, delegations []*delegate.Delegation) (ok int, failed int) {
	for _, delegation := range delegations {
		b, _ := json.Marshal(delegation) // this cannot produce an error

		_, err := c.SQS.SendMessageWithContext(ctx, &sqs.SendMessageInput{
			MessageBody: aws.String(string(b)),
			QueueUrl:    aws.String(c.SQSToken),
		})
		if err != nil {
			log.Printf("SQS ERROR: %s: %v", delegation.Subzone, err)
			failed++
		} else {
			ok++
		}
	}

	return ok, failed
}
