package delidangle_test

import (
	"encoding/json"
	"testing"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/route53"
	"github.com/aws/aws-sdk-go/service/sqs"
	gomock "github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
)

func TestScannerHandler(t *testing.T) {
	t.Parallel()
	assert := assert.New(t)
	ctx := aws.BackgroundContext()

	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	c := getMocks(mockCtrl)
	returns := &route53.ListResourceRecordSetsOutput{
		ResourceRecordSets: []*route53.ResourceRecordSet{{
			Type: aws.String(route53.RRTypeNs),
			Name: aws.String("subzoon.main.zone"),
			ResourceRecords: []*route53.ResourceRecord{
				{Value: aws.String("ns1")},
				{Value: aws.String("ns2")},
				{Value: aws.String("ns3")},
				{Value: aws.String("ns4")},
			},
		}},
	}
	do := func(ctx aws.Context, input *route53.ListResourceRecordSetsInput,
		fn func(*route53.ListResourceRecordSetsOutput, bool) bool, opts ...request.Option) {
		fn(returns, false)
	}

	c.MockDelegator.EXPECT().ListResourceRecordSetsPagesWithContext(ctx, &route53.ListResourceRecordSetsInput{
		HostedZoneId: aws.String(c.R53.ZoneID),
		MaxItems:     aws.String("300"),
	}, gomock.Any()).Return(nil).Do(do)
	c.MockQueuer.EXPECT().SendMessageWithContext(ctx, gomock.Any())
	assert.Nil(c.ScannerHandler(ctx), "no error should be generated for this request")

	c.MockDelegator.EXPECT().ListResourceRecordSetsPagesWithContext(ctx, &route53.ListResourceRecordSetsInput{
		HostedZoneId: aws.String(c.R53.ZoneID),
		MaxItems:     aws.String("300"),
	}, gomock.Any()).Return(nil).Do(do)
	c.MockQueuer.EXPECT().SendMessageWithContext(ctx, gomock.Any()).Return(nil, errTest)
	assert.Nil(c.ScannerHandler(ctx), "sqs errors are not returned, only logged")

	c.MockDelegator.EXPECT().ListResourceRecordSetsPagesWithContext(ctx, gomock.Any(), gomock.Any()).Return(errTest).Do(do)
	assert.ErrorIs(c.ScannerHandler(ctx), errTest, "the error must be returned.")

	c.R53.ZoneName = ""
	//
	c.MockDelegator.EXPECT().GetHostedZoneWithContext(ctx, gomock.Any(), []request.Option{}).Return(nil, errTest)
	assert.ErrorIs(c.ScannerHandler(ctx), errTest, "the error getting our own zone name must be returned.")
}

func TestWriteSQS(t *testing.T) { // nolint: funlen
	t.Parallel()
	assert := assert.New(t)
	ctx := aws.BackgroundContext()

	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	c := getMocks(mockCtrl)

	delegations := []*delegate.Delegation{{
		Subzone:     "sub.zone.com.",
		TTL:         aws.Int64(9999),
		Nameservers: nil,
	}, {
		Subzone:     "sub2.zone.com.",
		TTL:         aws.Int64(9999),
		Nameservers: nil,
	}}

	// First
	b0, err := json.Marshal(delegations[0])
	assert.Nil(err, "there must be no error while marshaling data")
	//
	b1, err := json.Marshal(delegations[1])
	assert.Nil(err, "there must be no error while marshaling data")
	//
	c.MockQueuer.EXPECT().SendMessageWithContext(ctx, gomock.Any()).Do(
		func(_ aws.Context, input *sqs.SendMessageInput) {
			assert.Equal(c.SQSToken, *input.QueueUrl, "the sqs url (token) was not used correctly")
			assert.Equal(string(b0), *input.MessageBody)
		},
	)
	c.MockQueuer.EXPECT().SendMessageWithContext(ctx, gomock.Any()).Do(
		func(_ aws.Context, input *sqs.SendMessageInput) {
			assert.Equal(c.SQSToken, *input.QueueUrl, "the sqs url (token) was not used correctly")
			assert.Equal(string(b1), *input.MessageBody)
		},
	)
	//
	ok, failed := c.WriteSQS(ctx, delegations)
	//
	assert.Equal(len(delegations), ok, "every message should be OK")
	assert.Equal(0, failed, "there must be no failed messages")

	// Next, send in 1 error.
	c.MockQueuer.EXPECT().SendMessageWithContext(ctx, gomock.Any()).Return(nil, nil)
	c.MockQueuer.EXPECT().SendMessageWithContext(ctx, gomock.Any()).Return(nil, errTest)
	//
	ok, failed = c.WriteSQS(ctx, delegations)
	//
	assert.Equal(1, ok, "1 of 2 messages were ok")
	assert.Equal(1, failed, "1 of 2 messages failed")

	// Last, send in 2 errors.
	c.MockQueuer.EXPECT().SendMessageWithContext(ctx, gomock.Any()).Return(nil, errTest)
	c.MockQueuer.EXPECT().SendMessageWithContext(ctx, gomock.Any()).Return(nil, errTest)
	//
	ok, failed = c.WriteSQS(ctx, delegations)
	//
	assert.Equal(0, ok, "no messages were ok")
	assert.Equal(len(delegations), failed, "all messages failed")

	// Catch emptyness
	ok, failed = c.WriteSQS(ctx, []*delegate.Delegation{})
	//
	assert.Equal(0, ok, "no messages were ok")
	assert.Equal(0, failed, "no messages failed")
}
