package delidangle

import (
	"context"
	"encoding/json"
	"fmt"
	"log"
	"os"
	"strings"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/miekg/dns"
)

// FailureData tracks failures from our name servers so we can send them to an SNS Topic.
type FailureData struct {
	Zone    string            `json:"zone"`
	Servers []*string         `json:"nameservers"`
	OK      int               `json:"ok_answers"`
	Errors  map[string]string `json:"errors"` // len == NotOK
	Deleted bool              `json:"deleted"`
	Safe    bool              `json:"safe"`
	Link    string            `json:"link"`
}

// Custom Errors.
var (
	ErrEmptyPayload     = fmt.Errorf("no records found in payload")
	ErrZoneOrServersNil = fmt.Errorf("zone or name servers in message are empty")
)

// WorkerHandler processes an SQS payload containing a zone name and its name servers.
func (c *Config) WorkerHandler(ctx context.Context, sqsEvent events.SQSEvent) error {
	if len(sqsEvent.Records) == 0 {
		return fmt.Errorf("invalid request: %w: %v", ErrEmptyPayload, sqsEvent)
	}

	for _, message := range sqsEvent.Records {
		delegation := &delegate.Delegation{
			// These are only shown if the delegation is deleted. They show up in the comment.
			AccountID: "unknown",
			ZoneID:    "unknown",
		}

		err := json.Unmarshal([]byte(message.Body), delegation)
		if err != nil {
			return fmt.Errorf("invalid record: %w: %v", err, message)
		} else if delegation.Subzone == "" || len(delegation.Nameservers) == 0 || delegation.TTL == nil {
			return fmt.Errorf("invalid record: %w: %v", ErrZoneOrServersNil, message)
		}

		data := c.IsBroken(ctx, delegation)

		if split := strings.Split(message.EventSourceARN, ":"); len(split) >= 5 { //nolint:gomnd
			data.Link = "https://isengard.amazon.com/federate?account=" + split[4] +
				"&role=admin&destination=route53%2Fv2%2Fhostedzones%23ListRecordSets%2F" + c.R53.ZoneID
		}

		if data.Deleted || (os.Getenv("NOISYSLACK") == "true" && len(data.Errors) > 0) {
			c.PublishSNS(data)
		}

		if !data.Deleted {
			continue
		} else if c.SafeMode {
			log.Printf("[BROKEN] %s delegation deleted! (safe mode) NSs: %s", delegation.Subzone, delegation.Nameservers)
			continue
		}

		log.Printf("[BROKEN] %s delegation deleted! NSs: %s", delegation.Subzone, delegation.Nameservers)

		if err := c.R53.Delete(ctx, delegation); err != nil { // none of the servers responded.
			log.Printf("[R53 ERROR] deleting delegation for %s: ChangeResourceRecordSets(): %v", delegation.Subzone, err)
		}
	}

	return nil
}

// IsBroken loops through each queue item and dispatches them for checking.
// Return FailureData.Deleted as false to delete the delegation. Return true to keep it.
func (c *Config) IsBroken(ctx context.Context, delegation *delegate.Delegation) *FailureData {
	log.Printf("[INFO] Checking %s, Nameservers: %s", delegation.Subzone, delegation.Nameservers)

	f := &FailureData{
		Servers: delegation.Nameservers.Slice(),
		Zone:    delegation.Subzone,
		Errors:  make(map[string]string),
		Deleted: false,
		Safe:    c.SafeMode,
		OK:      0,
	}

	for _, ns := range delegation.Nameservers {
		switch a, rtt, err := c.DNS.ExchangeContext(ctx, &dns.Msg{Question: []dns.Question{{
			Name:   delegation.Subzone,
			Qtype:  dns.TypeNS,
			Qclass: dns.ClassINET,
		}}}, *ns.Value+DNSPort); {
		case err != nil:
			f.Errors[*ns.Value] = "dns error: " + err.Error()
			log.Printf("[DNS ERROR] looking up %s @ %s: %v (rtt: %v)", delegation.Subzone, *ns.Value, err, rtt)
			// We don't continue processing if a server didn't respond.
			return f
		case a == nil || len(a.Answer) == 0:
			// a is never nil, but a broken delegation has an empty Answer.
			log.Printf("[DNS ERROR] server has no answer for %s: %s (rtt: %v)", delegation.Subzone, *ns.Value, rtt)
			f.Errors[*ns.Value] = "no answer" //nolint:wsl
		case len(a.Answer) != len(delegation.Nameservers):
			log.Printf("[DNS ERROR] server has wrong answer count for %s: %s (%d != %d)",
				delegation.Subzone, *ns.Value, len(a.Answer), len(delegation.Nameservers))
			f.Errors[*ns.Value] = "wrong answer count" //nolint:wsl
		case !serversMatch(delegation.Nameservers, a.Answer):
			log.Printf("[DNS ERROR] server has wrong NS answers for %s: %s (possible hijack), %d answers: %v, expected: %v",
				delegation.Subzone, *ns.Value, len(a.Answer), a.Answer, delegation.Nameservers)
			f.Errors[*ns.Value] = "wrong NS answers" //nolint:wsl
		default:
			log.Printf("[DNS ANSWER] %s @ %s: NS records: %d (rtt: %v)", delegation.Subzone, *ns.Value, len(a.Answer), rtt)
			f.OK++
		}
	}

	f.Deleted = f.OK == 0

	// return false if none of the servers are ok.
	return f
}

// PublishSNS sends a mesage to an SNS topic.
func (c *Config) PublishSNS(msg interface{}) {
	b, err := json.Marshal(msg)
	if err != nil {
		log.Printf("[SNS ERROR] json marshal: %v", err)

		return
	}

	switch o, err := c.SNS.Publish(&sns.PublishInput{
		Message:  aws.String(string(b)),
		TopicArn: aws.String(c.SNSToken),
	}); {
	case err != nil:
		log.Printf("[SNS ERROR] %s: failed: %v", c.SNSToken, err)
	case o == nil || o.MessageId == nil || *o.MessageId == "":
		log.Printf("[SNS ERROR] no message ID returned!")
	default:
		log.Printf("[SNS SENT] %s: Message ID: %s, Bytes: %d", c.SNSToken, *o.MessageId, len(b))
	}
}

func serversMatch(realNSs delegate.NameServers, answ []dns.RR) bool {
	for _, ns := range answ {
		if s := ns.(*dns.NS).Ns; s != "" && !realNSs.Contains(s) {
			return false
		}
	}

	return true
}
