package delidangle_test

import (
	"encoding/json"
	"testing"
	"time"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"code.justin.tv/awsi/twitch-a2z-com/pkg/delidangle"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/route53"
	gomock "github.com/golang/mock/gomock"
	"github.com/miekg/dns"
	"github.com/stretchr/testify/assert"
)

func TestIsBroken(t *testing.T) {
	t.Parallel()
	assert := assert.New(t)
	ctx := aws.BackgroundContext()

	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	c := getMocks(mockCtrl)
	delegation := &delegate.Delegation{
		Subzone: "sub.zone.com.",
		TTL:     aws.Int64(9999),
		Nameservers: []*route53.ResourceRecord{
			{Value: aws.String("ns1.")},
			{Value: aws.String("ns2.")},
			{Value: aws.String("ns3.")},
			{Value: aws.String("ns4.")},
		},
	}
	// This is a fake, but valid dns NS answer.
	// We do not currently _inspect_ the answer, only check that there _is_ one.
	// If we decide to actually check the SOA record, we can amend this test
	// to include real SOA values in the &dns.NS{} structure.
	returns := &dns.Msg{Answer: []dns.RR{
		&dns.NS{Ns: "ns2."},
		&dns.NS{Ns: "ns3."},
		&dns.NS{Ns: "ns4."},
		&dns.NS{Ns: "ns1."},
	}}

	// First, drop an error on second lookup. ie. like the dns server is down.
	v1 := *delegation.Nameservers[0].Value + delidangle.DNSPort
	v2 := *delegation.Nameservers[1].Value + delidangle.DNSPort
	///
	c.MockResolver.EXPECT().ExchangeContext(ctx, gomock.Any(), v1).Return(returns, time.Hour, nil)
	c.MockResolver.EXPECT().ExchangeContext(ctx, gomock.Any(), v2).Return(nil, time.Hour, errTest)

	d := c.IsBroken(ctx, delegation)
	assert.False(d.Deleted, "an error during a lookup must return false")

	// Next, make sure each name server does a lookup, and it returns false (not broken).
	for i := range delegation.Nameservers {
		v := *delegation.Nameservers[i].Value + delidangle.DNSPort
		c.MockResolver.EXPECT().ExchangeContext(ctx, gomock.Any(), v).Return(returns, time.Hour, nil)
	}
	///
	d = c.IsBroken(ctx, delegation)
	assert.False(d.Deleted, "all the lookups were OK, so this should be true")

	// Last, make sure each name server does an empty lookup, and it returns true (broken).
	for i := range delegation.Nameservers {
		v := *delegation.Nameservers[i].Value + delidangle.DNSPort
		c.MockResolver.EXPECT().ExchangeContext(ctx, gomock.Any(), v).Return(&dns.Msg{}, time.Hour, nil)
	}
	///
	d = c.IsBroken(ctx, delegation)
	assert.True(d.Deleted, "all the lookups were OK, so this should be true")
}

// This also tests the delete function.
func TestWorkerHandler(t *testing.T) { //nolint: funlen
	t.Parallel()
	assert := assert.New(t)
	ctx := aws.BackgroundContext()

	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	c := getMocks(mockCtrl)
	input := events.SQSEvent{}
	delegation := &delegate.Delegation{
		Subzone: "sub.zone.com.",
		TTL:     aws.Int64(9999),
		Nameservers: []*route53.ResourceRecord{
			{Value: aws.String("ns1.")},
			{Value: aws.String("ns2.")},
			{Value: aws.String("ns3.")},
			{Value: aws.String("ns4.")},
		},
	}

	// XXX: fix this.
	c.MockNotifier.EXPECT().Publish(gomock.Any()).AnyTimes()

	// Test all our error conditions, before testing delete.
	assert.ErrorIs(c.WorkerHandler(ctx, input), delidangle.ErrEmptyPayload, "the empty payload must produce an error")
	///
	input = events.SQSEvent{Records: []events.SQSMessage{{Body: "garbage"}}}
	assert.Error(c.WorkerHandler(ctx, input), "the junk payload must produce an error")
	///
	input = events.SQSEvent{Records: []events.SQSMessage{{Body: "{}"}}}
	assert.ErrorIs(c.WorkerHandler(ctx, input), delidangle.ErrZoneOrServersNil, "the empty payload must produce an error")

	// Test an OK delegation.
	body, _ := json.Marshal(delegation)
	input = events.SQSEvent{Records: []events.SQSMessage{{Body: string(body)}}}
	returns := &dns.Msg{Answer: []dns.RR{
		&dns.NS{Ns: "ns2."},
		&dns.NS{Ns: "ns3."},
		&dns.NS{Ns: "ns4."},
		&dns.NS{Ns: "ns1."},
	}}
	///
	c.MockResolver.EXPECT().ExchangeContext(ctx, gomock.Any(), gomock.Any()).
		Times(len(delegation.Nameservers)).Return(returns, time.Hour, nil)
	assert.Nil(c.WorkerHandler(ctx, input), "all the servers had an answer, so there should be no error and no deletions")

	// Now test the delete procedure.
	do := func(ctx aws.Context, input *route53.ChangeResourceRecordSetsInput, _ ...request.Option) {
		assert.Equal(c.R53.ZoneID, *input.HostedZoneId, "wrong zone was updated")
		assert.Equal(route53.ChangeActionDelete, *input.ChangeBatch.Changes[0].Action, "wrong action taken")
		assert.Equal(delegation.Subzone, *input.ChangeBatch.Changes[0].ResourceRecordSet.Name, "wrong record updated")
		assert.Equal(delegation.TTL, input.ChangeBatch.Changes[0].ResourceRecordSet.TTL, "wrong record updated")
		assert.Equal(route53.RRTypeNs, *input.ChangeBatch.Changes[0].ResourceRecordSet.Type, "wrong record updated")
		assert.EqualValues(delegation.Nameservers,
			input.ChangeBatch.Changes[0].ResourceRecordSet.ResourceRecords, "wrong record updated")
	}

	c.MockResolver.EXPECT().ExchangeContext(ctx, gomock.Any(), gomock.Any()).Times(len(delegation.Nameservers))
	c.MockDelegator.EXPECT().ChangeResourceRecordSetsWithContext(ctx, gomock.Any(), gomock.Any()).Do(do)
	assert.Nil(c.WorkerHandler(ctx, input),
		"all the servers had an answer, and record deleted, so there should be no error")

	// Test invalid name servers returned (hijacked domain).
	c.MockResolver.EXPECT().ExchangeContext(ctx, gomock.Any(), gomock.Any()).Times(len(delegation.Nameservers)).Return(
		&dns.Msg{Answer: []dns.RR{
			&dns.NS{Ns: "ns8."},
			&dns.NS{Ns: "ns7."},
			&dns.NS{Ns: "ns10."},
			&dns.NS{Ns: "ns1."},
		}},
		time.Second, nil)
	c.MockDelegator.EXPECT().ChangeResourceRecordSetsWithContext(ctx, gomock.Any(), gomock.Any()).Do(do)
	assert.Nil(c.WorkerHandler(ctx, input),
		"all of the servers had a wrong answer, and record deleted, so there should be no error")

	// Test invalid name servers returned (hijacked domain).
	c.MockResolver.EXPECT().ExchangeContext(ctx, gomock.Any(), gomock.Any()).Times(len(delegation.Nameservers)).Return(
		&dns.Msg{Answer: []dns.RR{
			&dns.NS{Ns: "ns1."},
			&dns.NS{Ns: "ns2."},
			&dns.NS{Ns: "ns3."},
		}},
		time.Second, nil)
	c.MockDelegator.EXPECT().ChangeResourceRecordSetsWithContext(ctx, gomock.Any(), gomock.Any()).Do(do)
	assert.Nil(c.WorkerHandler(ctx, input),
		"one of the servers had a wrong answer count, and record deleted, so there should be no error")

	// And make sure delete errors don't crash (all they do is log)
	c.MockResolver.EXPECT().ExchangeContext(ctx, gomock.Any(), gomock.Any()).Times(len(delegation.Nameservers))
	c.MockDelegator.EXPECT().ChangeResourceRecordSetsWithContext(ctx, gomock.Any(), gomock.Any()).Return(nil, errTest)
	assert.Nil(c.WorkerHandler(ctx, input), "a delete error need not be surfaced")
}
