package metrics

//nolint:lll
//go:generate mockgen -destination=../mocks/mock_maestro.go -package=mocks code.justin.tv/awsi/twitch-a2z-com/pkg/metrics Maestro

import (
	"context"
	"log"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
)

// Maestro allows mocking our CloudWatch calls.
type Maestro interface {
	PutMetricDataWithContext(context.Context, *cloudwatch.PutMetricDataInput, ...request.Option) (*cloudwatch.PutMetricDataOutput, error) // nolint: lll
}

// Metrics is the data needed to use this package.
type Metrics struct {
	ZoneID    string
	Namespace string
	Svc       Maestro
}

// Send pushes a single metric into CloudWatch.
func (m *Metrics) Send(metricName, zoneName, accountID string) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()

	_, err := m.Svc.PutMetricDataWithContext(ctx, &cloudwatch.PutMetricDataInput{
		Namespace: aws.String(m.Namespace),
		MetricData: []*cloudwatch.MetricDatum{{
			MetricName: aws.String(metricName),
			Unit:       aws.String("None"),
			Value:      aws.Float64(1),
			// The Go test expects these in a specific order.
			Dimensions: []*cloudwatch.Dimension{{
				Name:  aws.String("subzone"),
				Value: aws.String(zoneName),
			}, {
				Name:  aws.String("accountid"),
				Value: aws.String(accountID),
			}, {
				Name:  aws.String("zoneid"),
				Value: aws.String(m.ZoneID),
			}},
		}},
	})
	if err != nil {
		log.Printf("ERROR: Writing CloudWatch Metric: %v", err)
	}
}
