// Package slackhook is two modules in one: One module provides slack webhook
// payload handling and the other has specific assumptions for the dangling
// delegation checker and route53 changes. It should and can be split.
package slackhook

import (
	"context"
	"encoding/json"
	"fmt"
	"os"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delidangle"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-sdk-go/service/route53"
)

// Errors returned by this package.
var (
	ErrUnknownPayload = fmt.Errorf("the input payload is unknown")
)

// New returns a slack config to send messages. Designed for Lambda.
func New() *Config {
	debug, _ := strconv.ParseBool(os.Getenv("SLACK_DEBUG"))

	return &Config{
		URL:     strings.TrimSpace(os.Getenv("SLACK_HOOK")),
		Channel: strings.TrimSpace(os.Getenv("SLACK_CHAN")),
		BotName: strings.TrimSpace(os.Getenv("SLACK_NAME")),
		Emoji:   strings.TrimSpace(os.Getenv("SLACK_EMOJI")),
		Debug:   debug,
	}
}

// Handler is a lambda handler to send notifications to Slack from an SNS topic.
func (c *Config) Handler(ctx context.Context, snsEvent events.SNSEvent) error {
	if c.URL == "" || c.Channel == "" {
		return nil
	}

	for _, m := range snsEvent.Records {
		c.Debugf("Incoming Message from %s: %s", m.EventSource, m.SNS.Message)

		if err := c.Notify(ctx, []byte(m.SNS.Message), &m.SNS); err != nil {
			return err
		}
	}

	return nil
}

// Notify tries to unmarshal the payload into known data types.
// If ones returns data then the corresponding procedure if called.
func (c *Config) Notify(ctx context.Context, data []byte, m *events.SNSEntity) error {
	var (
		v delidangle.FailureData
		u Route53Event
	)

	err := json.Unmarshal(data, &v)
	if err == nil && len(v.Errors) != 0 {
		return c.NotifyFailureData(ctx, &v, m)
	}

	err = json.Unmarshal(data, &u)
	if err == nil && len(u.Detail.RequestParameters.ChangeBatch.Changes) != 0 {
		return c.NotifyRoute53Changes(ctx, &u)
	}

	return ErrUnknownPayload
}

// NotifyFailureData process a custom delidangle FailureData payload.
// It turns the data into a beautiful Slack notification.
func (c *Config) NotifyFailureData(ctx context.Context, data *delidangle.FailureData, m *events.SNSEntity) error {
	region := "unknown"

	topic := strings.Split(m.TopicArn, ":")
	if len(topic) > 2 { //nolint:gomnd
		region = topic[3]
	}

	color := "#faa850" // orange
	zoneInfo := fmt.Sprintf("*Servers _OK_*: %d of %d\n*SNS Topic:* `%s`\n*Region:* `%s`\n*Errors:*",
		data.OK, len(data.Servers), topic[len(topic)-1], region)

	switch {
	case data.Deleted && data.Safe:
		color = "#b30c8c" // purplish
		zoneInfo = "Zone *DANGLING!* _(safe mode)_\n" + zoneInfo
	case data.Deleted:
		color = "#b30c12" // red
		zoneInfo = "Zone *DELETED!*\n" + zoneInfo
	default:
		zoneInfo = "Zone *NOT* Deleted\n" + zoneInfo
	}

	errs := []*Text{}
	for k, v := range data.Errors {
		errs = append(errs, &Text{Text: fmt.Sprintf("`%s`\n%s", k, v)})
	}

	header := &BlkHeader{Text: &Text{Text: "Zone: " + data.Zone}}
	info := &BlkSection{Text: &Text{Text: zoneInfo}}
	errors := &BlkSection{Fields: errs}

	if data.Link != "" {
		button := &BlkActions{Elements: []Element{&ElemButton{URL: data.Link, Text: Text{Text: "AWS Route53 Console"}}}}

		return c.Send(ctx, "", &Attachment{
			Color:  color,
			Blocks: []interface{}{header, info, errors, button},
		})
	}

	return c.Send(ctx, "", &Attachment{
		Color:  color,
		Blocks: []interface{}{header, info, errors},
	})
}

// NotifyRoute53Changes process a CloudWatch Eventbridge -> SNS notification for a BatchChange.
// In other words, it turns record changes in Route53 into Slack notifications.
func (c *Config) NotifyRoute53Changes(ctx context.Context, payload *Route53Event) error {
	if cbc := payload.Detail.RequestParameters.ChangeBatch.Comment; cbc != "" {
		payload.Detail.RequestParameters.ChangeBatch.Comment = "\n*Comment:* " + cbc
	}

	caller := strings.Split(payload.Detail.UserIdentity.PrincipalID, ":")
	if len(caller) > 1 {
		payload.Detail.UserIdentity.PrincipalID = caller[1]
	}

	var blocks []interface{}
	blocks = append(blocks,
		&BlkHeader{Text: &Text{Text: fmt.Sprintf("%s: %s",
			payload.Detail.EventName, payload.Detail.RequestParameters.HostedZoneID)}},
		&BlkSection{Text: &Text{Text: fmt.Sprintf("Route53 %s: _%s_\n*Caller:* `%s`\n*IP:* `%s`%s",
			payload.DetailType, payload.Detail.ResponseElements.ChangeInfo.Status,
			payload.Detail.UserIdentity.PrincipalID, payload.Detail.SourceIPAddress,
			payload.Detail.RequestParameters.ChangeBatch.Comment)}},
	)

	for _, change := range payload.Detail.RequestParameters.ChangeBatch.Changes {
		text := "\n"
		for _, f := range change.ResourceRecordSet.ResourceRecords {
			text += fmt.Sprintf("%s IN %s %d %s\n", change.ResourceRecordSet.Name,
				change.ResourceRecordSet.Type, change.ResourceRecordSet.TTL, f.Value)
		}

		block := &BlkSection{Text: &Text{Text: fmt.Sprintf("*%s Records:* ```%s```", change.Action, text)}}

		switch change.Action {
		case route53.ChangeActionCreate: // green
			blocks = append(blocks, &Attachment{Color: "#00cc00", Blocks: []interface{}{block}})
		case route53.ChangeActionDelete: // red
			blocks = append(blocks, &Attachment{Color: "#b30c12", Blocks: []interface{}{block}})
		case route53.ChangeActionUpsert: // blue (never happens)
			blocks = append(blocks, &Attachment{Color: "#00cc8b", Blocks: []interface{}{block}})
		}
	}

	return c.Send(ctx, "", blocks...)
}

type Route53Event struct {
	Version    string    `json:"version"`
	ID         string    `json:"id"`
	DetailType string    `json:"detail-type"`
	Source     string    `json:"source"`
	Account    string    `json:"account"`
	Time       time.Time `json:"time"`
	Region     string    `json:"region"`
	Detail     struct {
		UserIdentity struct {
			Type        string `json:"type"`
			PrincipalID string `json:"principalId"`
			Arn         string `json:"arn"`
			AccountID   string `json:"accountId"`
			AccessKeyID string `json:"accessKeyId"`
		} `json:"userIdentity"`
		EventName         string `json:"eventName"`
		SourceIPAddress   string `json:"sourceIPAddress"`
		UserAgent         string `json:"userAgent"`
		RequestParameters struct {
			HostedZoneID string `json:"hostedZoneId"`
			ChangeBatch  struct {
				Comment string
				Changes []struct {
					Action            string `json:"action"`
					ResourceRecordSet struct {
						Name            string `json:"name"`
						Type            string `json:"type"`
						TTL             int    `json:"tTL"`
						ResourceRecords []struct {
							Value string `json:"value"`
						} `json:"resourceRecords"`
					} `json:"resourceRecordSet"`
				} `json:"changes"`
			} `json:"changeBatch"`
		} `json:"requestParameters"`
		ResponseElements struct {
			ChangeInfo struct {
				ID          string `json:"id"`
				Status      string `json:"status"`
				SubmittedAt string `json:"submittedAt"`
			} `json:"changeInfo"`
		} `json:"responseElements"`
	} `json:"detail"`
}
