package storage

//nolint:lll
//go:generate mockgen -destination=../mocks/mock_storer.go -package=mocks code.justin.tv/awsi/twitch-a2z-com/pkg/storage Storer

import (
	"bytes"
	"encoding/json"
	"fmt"
	"path"
	"strings"
	"time"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/s3"
)

// Various Statii.
const (
	Request = "Requested"
	Granted = "Granted"
	Deleted = "Deleted"
)

// Storer allows mocking our S3 calls.
type Storer interface {
	GetObjectWithContext(aws.Context, *s3.GetObjectInput, ...request.Option) (*s3.GetObjectOutput, error)
	PutObjectWithContext(aws.Context, *s3.PutObjectInput, ...request.Option) (*s3.PutObjectOutput, error)
}

// Storage is our main data struct that operates the library.
type Storage struct {
	Svc    Storer
	Prefix string
	Bucket string
}

// StorePayload is the payload saved to s3.
type StorePayload struct {
	ZoneName  string    `json:"zone_name"`
	AccountID string    `json:"account_id"`
	StackID   string    `json:"stack_id"`
	ZoneID    string    `json:"zone_id"`
	Updated   time.Time `json:"updated"`
	Status    string    `json:"status"`
}

const suffix = "json"

// Save writes a delegation status to s3.
func (s *Storage) Save(ctx aws.Context, delegation *delegate.Delegation, stackid, status string) error {
	payload := &StorePayload{
		ZoneName:  delegation.Subzone,
		AccountID: delegation.AccountID,
		StackID:   stackid,
		ZoneID:    delegation.ZoneID,
		Updated:   time.Now().UTC(),
		Status:    status,
	}

	err := s.SaveRaw(ctx, payload, path.Join(delegation.AccountID, delegation.Subzone))
	if err != nil {
		return fmt.Errorf("%s: %s failed: %w", delegation.Subzone, status, err)
	}

	return nil
}

// SaveRaw writes an interface{} to an s3key as json.
func (s *Storage) SaveRaw(ctx aws.Context, payload interface{}, s3key string) error {
	s3path := path.Join("/", s.Prefix, strings.TrimSuffix(s3key, ".")+".") + suffix

	storeBytes, err := json.Marshal(payload)
	if err != nil {
		return fmt.Errorf("marshaling payload: %w", err)
	}

	_, err = s.Svc.PutObjectWithContext(ctx, &s3.PutObjectInput{
		Bucket:             aws.String(s.Bucket),
		Key:                aws.String(s3path),
		ACL:                aws.String("private"),
		Body:               bytes.NewReader(storeBytes),
		ContentLength:      aws.Int64(int64(len(storeBytes))),
		ContentType:        aws.String("application/json"),
		ContentDisposition: aws.String("attachment"),
	})
	if err != nil {
		return fmt.Errorf("PutObjectWithContext: %w", err)
	}

	return nil
}

// Get fetches, unmarshals and returns an s3 storage object.
func (s *Storage) Get(ctx aws.Context, accountID, subzone string) (*StorePayload, error) {
	s3path := path.Join("/", s.Prefix, accountID, subzone) + suffix

	obj, err := s.Svc.GetObjectWithContext(ctx, &s3.GetObjectInput{
		Bucket: aws.String(s.Bucket),
		Key:    aws.String(s3path),
	})
	if err != nil {
		return nil, fmt.Errorf("fetching s3 grant failed @ %s: %w", s3path, err)
	}
	defer obj.Body.Close()

	store := &StorePayload{}

	// This error isn't wrapped because it's highly unlikely to occur.
	return store, json.NewDecoder(obj.Body).Decode(store)
}
