package storage_test

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"testing"
	"time"

	"code.justin.tv/awsi/twitch-a2z-com/pkg/delegate"
	"code.justin.tv/awsi/twitch-a2z-com/pkg/mocks"
	"code.justin.tv/awsi/twitch-a2z-com/pkg/storage"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
)

var errTest = fmt.Errorf("this is a test error")

func testGetPayload() (io.ReadCloser, *delegate.Delegation, *storage.StorePayload) {
	delegation := &delegate.Delegation{
		Subzone:   "captain2.testing.x2y.dev.",
		ZoneID:    "Z0590442MVU3P715PESI",
		AccountID: "386434559102",
	}
	payload := &storage.StorePayload{
		ZoneID:    delegation.ZoneID,
		ZoneName:  delegation.Subzone,
		AccountID: delegation.AccountID,
		StackID:   "arn:aws:cloudformation:us-west-2:386434559102:stack/capt/de86d940-354f-11eb-9439-06d22a8932c3",
		Updated:   time.Now().Add(-time.Hour),
		Status:    storage.Granted,
	}
	b, _ := json.Marshal(payload)

	return ioutil.NopCloser(bytes.NewReader(b)), delegation, payload
}

// func (s *Storage) Save(ctx aws.Context, delegation *delegate.Delegation, stackid, status string) error {

func TestSave(t *testing.T) {
	t.Parallel()
	assert := assert.New(t)
	ctx := aws.BackgroundContext()

	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	mockstore := mocks.NewMockStorer(mockCtrl)
	s := &storage.Storage{
		Bucket: "bucket-name",
		Prefix: "prefixered",
		Svc:    mockstore,
	}
	_, delegation, jsonPayload := testGetPayload()

	mockstore.EXPECT().PutObjectWithContext(ctx, gomock.Any()).Do(
		func(ctx aws.Context, obj *s3.PutObjectInput) {
			// Mock the internals of the method to inspect the values passed in.
			assert.Equal("/"+s.Prefix+"/"+jsonPayload.AccountID+"/"+jsonPayload.ZoneName+"json", *obj.Key)
			assert.Equal("private", *obj.ACL)
			assert.Equal(s.Bucket, *obj.Bucket)
			assert.InDelta(int64(268), *obj.ContentLength, 3) // this may change a few bytes depending on time.Now()
			assert.Equal("application/json", *obj.ContentType)
			assert.Equal("attachment", *obj.ContentDisposition)

			// Decode the body and verify the values.
			body := &storage.StorePayload{}
			_ = json.NewDecoder(obj.Body).Decode(body)

			assert.Equal(jsonPayload.ZoneName, body.ZoneName)
			assert.Equal(jsonPayload.AccountID, body.AccountID)
			assert.Equal(jsonPayload.StackID, body.StackID)
			assert.Equal(jsonPayload.ZoneID, body.ZoneID)
			assert.Equal(jsonPayload.Status, body.Status)
			assert.WithinDuration(time.Now(), body.Updated, time.Second)
		},
	).Return(nil, nil)
	assert.Nil(s.Save(ctx, delegation, jsonPayload.StackID, jsonPayload.Status))

	// Make sure an error gets returned.
	mockstore.EXPECT().PutObjectWithContext(ctx, gomock.Any()).Return(nil, errTest)
	assert.ErrorIs(s.Save(ctx, delegation, jsonPayload.StackID, jsonPayload.Status), errTest)
}

func TestGet(t *testing.T) {
	t.Parallel()
	assert := assert.New(t)
	ctx := aws.BackgroundContext()

	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	mockstore := mocks.NewMockStorer(mockCtrl)
	s := &storage.Storage{
		Bucket: "bucket-name",
		Prefix: "prefixered",
		Svc:    mockstore,
	}

	// This provides the "pretend" payload returned from S3.
	jsonReader, _, jsonPayload := testGetPayload()

	mockstore.EXPECT().GetObjectWithContext(ctx, &s3.GetObjectInput{
		Bucket: &s.Bucket,
		Key:    aws.String("/" + s.Prefix + "/accountID/sub.zone.json"),
	}).Return(&s3.GetObjectOutput{Body: jsonReader}, nil)

	payload, err := s.Get(ctx, "accountID", "sub.zone.")
	assert.Nil(err, "The payload should be returned, not at an error!")
	assert.Equal(jsonPayload.ZoneID, payload.ZoneID, "The returned payload may be invalid.")
	assert.Equal(jsonPayload.ZoneName, payload.ZoneName, "The returned payload may be invalid.")
	assert.Equal(jsonPayload.StackID, payload.StackID, "The returned payload may be invalid.")
	assert.Equal(jsonPayload.Status, payload.Status, "The returned payload may be invalid.")
	assert.Equal(jsonPayload.AccountID, payload.AccountID, "The returned payload may be invalid.")
	assert.WithinDuration(jsonPayload.Updated, payload.Updated, time.Millisecond, "The returned payload may be invalid.")

	mockstore.EXPECT().GetObjectWithContext(ctx, &s3.GetObjectInput{
		Bucket: &s.Bucket,
		Key:    aws.String("/" + s.Prefix + "/accountID/sub.zone.json"),
	}).Return(nil, errTest)

	payload, err = s.Get(ctx, "accountID", "sub.zone.")
	assert.ErrorIs(err, errTest, "The provided error must be returned.")
	assert.Nil(payload, "the payload must be nil when an error returns")
}
