data "aws_ecs_cluster" "main" {
  cluster_name = "${var.ecs_cluster_name}"
}

data "aws_iam_role" "ecs-registration" {
  name = "${var.ecs_cluster_name}-ecs-service"
}

// this will always fetch the latest ecs task definition.
data "aws_ecs_task_definition" "ecs-service" {
  task_definition = "${local.ecs_task_family}"
}

resource "aws_ecs_service" "ecs-service" {
  name            = "${local.ecs_service_name}"
  cluster         = "${data.aws_ecs_cluster.main.arn}"
  task_definition = "${data.aws_ecs_task_definition.ecs-service.family}:${data.aws_ecs_task_definition.ecs-service.revision}"
  desired_count   = 3
  iam_role        = "${data.aws_iam_role.ecs-registration.arn}"

  ordered_placement_strategy {
    type  = "binpack"
    field = "cpu"
  }

  load_balancer {
    target_group_arn = "${aws_alb_target_group.ecs-service.arn}"
    container_name   = "${local.load_balancer_container_name}"
    container_port   = "${local.load_balancer_container_port}"
  }

  depends_on = [
    "aws_alb.ecs-service",
    "aws_alb_listener.ecs-service",
  ]
}

resource "aws_alb_target_group" "ecs-service" {
  name     = "${local.prefix}"
  port     = "${local.load_balancer_container_port}"
  protocol = "HTTP"
  vpc_id   = "${var.vpc_id}"

  health_check {
    matcher = "200"
    path    = "/health"
  }

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_alb" "ecs-service" {
  name            = "${local.prefix}"
  internal        = true
  security_groups = ["${var.security_groups}"]
  subnets         = "${var.subnet_ids}"

  tags = {
    Service     = "${local.service_name}"
    Environment = "${var.env}"
  }
}

resource "aws_alb_listener" "ecs-service" {
  load_balancer_arn = "${aws_alb.ecs-service.arn}"
  port              = "443"
  protocol          = "HTTPS"
  ssl_policy        = "ELBSecurityPolicy-2015-05"
  certificate_arn   = "${var.acm_certificate_arn}"

  default_action {
    target_group_arn = "${aws_alb_target_group.ecs-service.arn}"
    type             = "forward"
  }
}

resource "aws_cloudwatch_log_group" "ecs-service" {
  name = "/twitch/${local.service_name}/${var.env}"

  tags = {
    Service     = "${local.service_name}"
    Environment = "${var.env}"
  }
}
