package envoy

import (
	"fmt"
	"log"
	"os/exec"
	"strings"
	"time"
)

const (
	DefaultNSCATimeout time.Duration = 5 * time.Second
	DefaultNSCAPath    string        = "/usr/sbin/send_nsca"
	nagiosCritical                   = 2
	nagiosUnknown                    = 3
	nagiosWarning                    = 1
	nagiosNormal                     = 0
)

var (
	lastPush          time.Time
	nagiosStatusCodes = map[Condition]int16{
		ConditionCritical: nagiosCritical,
		ConditionUnknown:  nagiosUnknown,
		ConditionWarning:  nagiosWarning,
		ConditionNormal:   nagiosNormal,
	}
)

type sendNSCAMsg struct {
	State   int16
	Host    string
	Service string
	Message string
}

func (m *sendNSCAMsg) String() string {
	elements := []string{m.Host, m.Service, fmt.Sprintf("%d", m.State), strings.Replace(m.Message, "\n", "\\n", -1)}
	return strings.Join(elements, "\t") + "\n"
}

func (e *Envoy) runPusher(conf *NagiosConf) {
	for {
		<-e.transitions

		if time.Since(lastPush) < e.windowLength {
			time.Sleep(e.windowLength - time.Since(lastPush))
		}

		lastPush = time.Now()
		snapshot := e.newStateSnapshots()

		nagiosCode, found := nagiosStatusCodes[snapshot.GlobalSnapshot.Current]
		if !found {
			nagiosCode = nagiosUnknown
		}

		msg := &sendNSCAMsg{
			State:   nagiosCode,
			Host:    conf.Node,
			Service: conf.Service,
			Message: snapshot.String(),
		}

		done := make(chan bool)
		go func() {
			timeoutArg := fmt.Sprintf("%.0f", conf.NSCATimeout.Seconds())
			cmd := exec.Command(conf.NSCAPath, "-H", conf.Host, "-p", conf.Port, "-to", timeoutArg)
			cmd.Stdin = strings.NewReader(msg.String())
			output, err := cmd.CombinedOutput()
			if err != nil {
				log.Printf("ENVOY ERROR: send_nsca %+v, output was %s", err, output)
				done <- true
				return
			}

			done <- true
		}()

		select {
		case <-done:
			continue
		case <-time.After(conf.NSCATimeout):
			log.Println("ENVOY ERROR: NSCA update failed due to timeout.")
			continue
		}
	}
}
