package dbopt

import (
	"encoding/json"
	"flag"
	"fmt"
	"net/http"
	"regexp"
	"strings"
	"sync"
	"time"
)

var (
	dboptURL = flag.String(
		"dbopt-url",
		"http://usher.justin.tv/dboption/all/",
		"URL pattern specifying where to query DB options from Usher")

	// DefaultClient is a default db opt client that can be shared between multiple clients
	DefaultClient = defaultClient()
)

func defaultClient() *Client {
	res := NewClient(10*time.Second, dboptURL)
	res.ErrorHandler(EnvoyErrorHandler())
	return res
}

type patternMgrCallbacks interface {
	transformPattern(pattern string) string
	skipPattern(pattenr string) bool
	mergedPatterns()
}

type patternMgr struct {
	patCB               patternMgrCallbacks
	patSet              map[string]bool
	patterns            []string
	transformedPatterns []string
}

type listener struct {
	patternMgr
	re *regexp.Regexp
}

// Client is used to query db option values. It distributes updates to dboptions
// to channels that have been registered as caring about values with those
// patterns via RegisterChannel().
type Client struct {
	sync.RWMutex
	patternMgr

	patRE *regexp.Regexp

	baseURL    string
	url        string
	httpClient http.Client
	ticker     *time.Ticker
	closeC     chan bool

	listeners     map[chan<- Update]*listener
	errorHandlers map[chan<- error]bool
	stopped       bool
}

// Update describes an updated Value for the dboption with the specified Name.
type Update struct {
	Name  string
	Value interface{}
}

func fallback(ps ...interface{}) interface{} {
	for _, p := range ps {
		if p != nil {
			return p
		}
	}
	return nil
}

// NewClient creates a new DB option client with the specified frequency
// between successive fetches of DB values, base url (which specifies where
// to fetch DB options from), and patterns.
//
// If base url is not specified, the client will default to the value specified
// via the --dbopt-url command line flag.
//
// patterns are a list of values to fetch; these can include * as a wildcard.
func NewClient(freq time.Duration, url *string, patterns ...string) *Client {
	res := &Client{
		patternMgr:    patternMgr{patSet: make(map[string]bool)},
		baseURL:       *fallback(url, dboptURL).(*string),
		httpClient:    http.Client{Timeout: freq},
		ticker:        time.NewTicker(freq),
		closeC:        make(chan bool),
		listeners:     make(map[chan<- Update]*listener),
		errorHandlers: make(map[chan<- error]bool),
	}
	res.patternMgr.patCB = res
	res.mergePatterns(patterns...)

	// This stops RegisterChannel() from adding the catchall wildcard, which is
	// what we want. ("*" in RegisterChannel receives all updates, but doesn't pull
	// them all from Usher.)
	res.patSet["*"] = true

	go res.loop()
	return res
}

type valueMap map[string]interface{}

func (s *Client) loop() {
	for {
		select {
		case <-s.closeC:
			return

		case <-s.ticker.C:
			go s.update()
		}
	}
}

func (s *Client) update() {
	s.RLock()
	defer s.RUnlock()

	for name, val := range s.fetchOptions() {
		s.sendUpdate(name, val)
	}
}

// FetchPatterns adds the specified patterns to the list of patterns to
// fetch from Usher.
func (s *Client) FetchPatterns(patterns ...string) {
	s.mergePatterns(patterns...)
}

func (s *Client) fetchOptions() valueMap {
	s.RLock()
	defer s.RUnlock()

	resp, err := s.httpClient.Get(s.url)

	defer func() {
		if resp.Body != nil {
			resp.Body.Close()
		}
	}()

	var maps []valueMap
	var result valueMap

	if err != nil {
		err = fmt.Errorf("Error during HTTP get: %#v", err)
	} else {
		dec := json.NewDecoder(resp.Body)
		dec.UseNumber()
		err = dec.Decode(&maps)
	}

	if err != nil {
		err = fmt.Errorf("Error during JSON decode: %#v", err)
	} else {
		if len(maps) != 1 {
			err = fmt.Errorf("Expected 1 maps in response, but got %d", len(maps))
		}
	}

	if err == nil {
		result = maps[0]
	}

	s.err(err)
	return result
}

func (s *Client) sendUpdate(name string, val interface{}) {
	s.RLock()
	defer s.RUnlock()

	update := Update{name, val}

	for ch, lsn := range s.listeners {
		if lsn.wantsName(name) {
			select {
			case ch <- update:
				/* Good */
			default:
				/* Also good */
			}
		}
	}
}

func (s *Client) err(err error) {
	s.RLock()
	defer s.RUnlock()

	for c := range s.errorHandlers {
		select {
		case c <- err:
			/* Good */
		default:
			/* Also good */
		}
	}
}

// RegisterChannel registers the specified channel to receive updates for the
// specified patterns. Multiple channels can receive updates for the same value.
//
// If the Client is not currently querying the specified pattern it will
// start querying it in future updates. This does not apply for the special
// pattern string "*". If you specify "*" you will get updates for all patterns
// the client is currently querying; the client will *not* implicitly start
// querying all dboption values known to Usher for "*".
//
// Client does not block on sending updates. If an update channel is filled,
// updates will get dropped on the floor.
func (s *Client) RegisterChannel(c chan<- Update, patterns ...string) {
	s.Lock()
	defer s.Unlock()

	lis, ok := s.listeners[c]
	if !ok {
		lis = &listener{patternMgr: patternMgr{patSet: make(map[string]bool)}}
		lis.patternMgr.patCB = lis
		s.listeners[c] = lis
	}

	lis.mergePatterns(patterns...)
	s.mergePatterns(patterns...)
}

// ErrorHandler registers an error handler. Errors while fetching or decoding
// DB options are send to all channels registered via ErrorHandler().
//
// Client does not block on sending errors. If an error handling channel is
// filled, errors will get dropped on the floor.
func (s *Client) ErrorHandler(c chan<- error) {
	s.Lock()
	defer s.Unlock()
	s.errorHandlers[c] = true
}

// Stop stops the client. No more DB option updates will be delivered.
func (s *Client) Stop() {
	s.Lock()
	defer s.Unlock()

	if !s.stopped {
		s.ticker.Stop()
		close(s.closeC)
		s.stopped = true
		s.listeners = map[chan<- Update]*listener{}
	}
}

func patternToRE(pattern string) *regexp.Regexp {
	return &regexp.Regexp{}
}

func (s *patternMgr) mergePatterns(patterns ...string) {
	for _, pattern := range patterns {
		if _, ok := s.patSet[pattern]; !ok {
			if !s.patCB.skipPattern(pattern) {
				s.patSet[pattern] = true
				s.addPattern(pattern)
			}
		}
	}

	s.patCB.mergedPatterns()
}

func (s *patternMgr) addPattern(pattern string) {
	s.patterns = append(s.patterns, pattern)
	s.transformedPatterns = append(s.patterns, s.patCB.transformPattern(pattern))
}

func (s *patternMgr) transformPattern(pattern string) string { return pattern }
func (s *patternMgr) mergedPatterns()                        {}

func (s *Client) mergedPatterns() {
	s.url = s.baseURL + strings.Join(s.patterns, ",") + ".json"
	s.patRE = regexp.MustCompile("^" + strings.Join(s.transformedPatterns, "|") + "$")
}

func (s *Client) skipPattern(pattern string) bool {
	if s.patRE == nil {
		return false
	}
	// Don't add new patterns that are already matched by older ones
	// This means we won't request foo.bar if we already request foo.*
	return s.patRE.MatchString(pattern)
}

func (s *listener) transformPattern(pattern string) string {
	pat := regexp.QuoteMeta(pattern)
	return strings.Replace(pat, "\\*", ".*", -1)
}

func (s *listener) mergedPatterns() {
	s.re = regexp.MustCompile("^" + strings.Join(s.transformedPatterns, "|") + "$")
}

func (s *listener) wantsName(name string) bool {
	if s.re == nil {
		return false
	}
	return s.re.MatchString(name)
}

func (s *listener) skipPattern(pattern string) bool { return s.wantsName(pattern) }
