require './redshift'
require './templates'

class QueryRunner
  def initialize(out: STDOUT, err: STDERR)
    @all_good = true
    @templates = Templates.new
    @out, @err = out, err
  end

  def all_good?()
    @all_good
  end

  def connect(**args)
    raise "already connected" if @conn
    @conn = timed("connecting") do
      redshift_connect(**args)
    end

    return nil
  end

  def run_query(query_file,
      label: query_file,
      query_args: {},
      dump_all: false)
    qry = @templates.render_file(query_file, query_args)

    result = timed(label) do
      redshift_query(@conn, qry)
    end

    dump_result(label, result) if dump_all

    return result
  end

  def check_query(*args, label: query_file, constraints: {}, **kwargs)
    result = run_query(*args, label: label, **kwargs)

    problems = check_result_for_problems(result, constraints)
    return true if problems.empty?

    report_problems(label, problems)
    return false
  end

  private
  def check_result_for_problems(result, constraints, label: nil)
    problems = []
    result.each do |row|
      row_problems = []
      constraints.each do |field, constraint|
        field = field.to_s
        unless constraint === row[field]
          row_problems << [field, constraint]
        end
      end
      unless row_problems.empty?
        problems << [row, row_problems]
        @all_good = false
      end
    end

    return problems
  end

  def report_problems(label, problems)
    @err.puts "unhappy with #{label}:"
    problems.each do |row, row_problems|
      @err.puts "  * bad row: `#{row.inspect}`"
      row_problems.each do |field, constraint|
        @err.puts "      * `#{field}` should match `#{constraint.inspect}` but is `#{row[field].inspect}`"
      end
    end
    @err.puts
    @err.flush

    return nil
  end

  def timed(name = nil)
    prefix = "#{name}: " if name
    start_time = Time.now
    @out.puts "#{prefix}start at #{start_time}"
    @out.flush
    begin
      return yield
    ensure
      end_time = Time.now
      @out.puts "#{prefix}end at #{end_time} after #{end_time - start_time}s"
      @out.flush
    end
  end

  def dump_result(label, r)
    @out.puts "results for #{label}:"
    @out.puts r.fields.join("\t")
    r.values.each do |row|
      @out.puts row.join("\t")
    end
    @out.puts
    @out.flush
  end
end

def require_env(name)
  full_name = "VIEWCOUNTS_ACE_ALERTS_#{name}"
  unless v = ENV[full_name]
    raise "missing env var: #{full_name}"
  end

  return v
end

cluster = require_env("REDSHIFT_CLUSTER")
user = require_env("REDSHIFT_USER")

q = QueryRunner.new
q.connect(cluster_identifier: cluster, user: user)

q.check_query("queries/check-channel-concurrents.sql",
  label: "recent channel_concurrents",
  query_args: {
    time_from: "1 hour",
    time_to: "0 hours",
    agg_interval: "5", # minutes
  },
  constraints: {
    avg_diff: (0..4),
  }, dump_all: true)

# we're thinning out data older than 1 hour every 2 hours, which takes
# single-digit amounts of minutes, and we'd like to avoid including
# both thinned and non-thinned data in the same statistic, so let's
# add some safety margin
delay = (1 + 2) * 60 + 10
q.check_query("queries/check-channel-concurrents.sql",
  label: "older channel_concurrents",
  query_args: {
    time_from: "#{delay + 60 * 3} minutes",
    time_to: "#{delay} minutes",
    # thinned-out data means we keep one minute out of five, so we want
    # to aggregate over some more time so we get more than one datapoint
    agg_interval: "15", # minutes
  },
  constraints: {
    avg_diff: (0..6),
  }, dump_all: true)

exit -1 unless q.all_good?
