package server

import (
	"fmt"
	"log"
	"time"

	"code.justin.tv/bootcamp/aws-example/backend"
	example "code.justin.tv/bootcamp/aws-example/rpc/aws_example_service"

	"golang.org/x/net/context"
)

type Server struct {
	Backend backend.BackendIface
	Noop    bool
}

func New(config *ServerConfig) *Server {
	return &Server{
		Backend: config.Backend,
		Noop:    config.Noop,
	}
}

func (s *Server) UploadAsset(ctx context.Context, asset *example.Asset) (*example.AssetMetadata, error) {
	if s.Noop {
		return &example.AssetMetadata{}, nil
	}

	var err error
	md := &example.AssetMetadata{
		Id:        asset.Id,
		Createdat: time.Now().UTC().Format(time.RFC3339),
	}
	err = s.Backend.UploadToS3(asset.Id, asset.Blob)
	if err != nil {
		md.Error = true
		md.Cause = err.Error()
	}

	err = s.Backend.WriteToDDB(md)
	if err != nil {
		log.Print("Id: %s, %v", asset.Id, err)
		err = fmt.Errorf("Internal server error with upload %s", asset.Id)
	}
	return md, err
}

func (s *Server) GetAssetMetadata(ctx context.Context, asset *example.Asset) (*example.AssetMetadata, error) {
	if s.Noop {
		return &example.AssetMetadata{}, nil
	}
	md, err := s.Backend.GetFromDDB(asset.Id)
	if err != nil {
		err = fmt.Errorf("Internal server error with upload %s", asset.Id)
	}

	return md, err
}

func (s *Server) DeleteAsset(ctx context.Context, asset *example.Asset) (*example.AssetMetadata, error) {
	if s.Noop {
		return &example.AssetMetadata{}, nil
	}

	md, err := s.Backend.GetFromDDB(asset.Id)
	if err == nil {
		e := s.Backend.RemoveFromS3(asset.Id)
		if e != nil {
			md.Error = true
			md.Cause = err.Error()
		} else {
			md.Error = false
			md.Cause = ""
			md.Deletedat = time.Now().UTC().Format(time.RFC3339)
		}
		err = s.Backend.WriteToDDB(md)
		if err != nil {
			log.Printf("Id: %s, %v", asset.Id, err)
			err = fmt.Errorf("Internal server error with upload %s", asset.Id)
		}
	}

	return md, err
}
