package main

import (
	"strconv"
	"strings"
)

type twirp struct {
	// Map to record whether we've built each package
	pkgs map[string]*FileDescriptor
	gen  *Generator
}

var pkgs map[string]string

func init() {
	RegisterPlugin(new(twirp))
	pkgs = make(map[string]string)
}

func (t *twirp) Name() string { return "twirp" }

func (t *twirp) Init(gen *Generator) {
	t.pkgs = make(map[string]*FileDescriptor)
	t.gen = gen
}

func (t *twirp) Generate(file *FileDescriptor) {
	if len(file.svc) == 0 {
		return
	}

	t.GenerateUtils(file)

	// For each service, generate client stubs and server
	for i, service := range file.svc {
		t.generateService(file, service, i)
	}
}

func (t *twirp) GenerateImports(file *FileDescriptor) {
	if len(file.svc) == 0 {
		return
	}

	t.P(`import httplib`)
	t.P(`import json`)
	t.P(`import urllib2`)
	t.P(`from google.protobuf import symbol_database as _symbol_database`)
	t.P()

	protoFile := t.gen.file.baseFileName() + "_pb2"
	t.P(`import `, protoFile)
	t.P()
}

func (t *twirp) GenerateUtils(file *FileDescriptor) {
	t.P(`_sym_db = _symbol_database.Default()`)
	t.P()
	t.P(`class TwirpException(httplib.HTTPException):`)
	t.P(`    def __init__(self, code, message, meta):`)
	t.P(`        self.code = code`)
	t.P(`        self.message = message`)
	t.P(`        self.meta = meta`)
	t.P()
	t.P(`    @classmethod`)
	t.P(`    def from_http_err(cls, err):`)
	t.P(`        try:`)
	t.P(`            jsonerr = json.load(err)`)
	t.P(`            code = jsonerr["code"]`)
	t.P(`            msg = jsonerr["msg"]`)
	t.P(`            meta = jsonerr.get("meta")`)
	t.P(`            if meta is None:`)
	t.P(`                meta = {}`)
	t.P(`        except:`)
	t.P(`            code = "internal"`)
	t.P(`            msg = "Error from intermediary with HTTP status code {} {}".format(`)
	t.P(`                err.code, httplib.responses[err.code],`)
	t.P(`            )`)
	t.P(`        return cls(code, msg, meta)`)
	t.P()
}

// P forwards to g.gen.P, which prints output.
func (t *twirp) P(args ...string) { t.gen.P(args...) }

// Big header comments to makes it easier to visually parse a generated file.
func (t *twirp) sectionComment(sectionTitle string) {
	t.P()
	t.P(`# `, strings.Repeat("=", len(sectionTitle)))
	t.P(`# `, sectionTitle)
	t.P(`# `, strings.Repeat("=", len(sectionTitle)))
	t.P()
}

func (t *twirp) generateService(file *FileDescriptor, service *ServiceDescriptor, index int) {
	servName := serviceName(service)

	t.sectionComment(servName + ` Protobuf Client`)
	t.generateProtobufClient(file, service)
}

func (t *twirp) generateProtobufClient(file *FileDescriptor, service *ServiceDescriptor) {
	t.P(`class `, clientName(service), `(object):`)
	if t.gen.commentsAvailable(service.path) {
		t.P(`    """`)
		t.gen.PrintComments(service.path, `    `)
		t.P(`    """`)
		t.P()
	}
	t.P(`    def __init__(self, server_address):`)
	t.P(`        """Creates a new client for the `, serviceName(service), ` service.`)
	t.P()
	t.P(`        Args:`)
	t.P(`            server_address: The address of the server to send requests to, in`)
	t.P(`                the full protocol://host:port form.`)
	t.P(`        """`)
	t.P(`        self.__target = server_address`)
	t.P(`        self.__service_name = `, strconv.Quote(fullServiceName(file, service)))
	t.P()
	t.P(`    def __make_request(self, body, full_method):`)
	t.P(`        req = urllib2.Request(`)
	t.P(`            url=self.__target + "/twirp" + full_method,`)
	t.P(`            data=body,`)
	t.P(`            headers={"Content-Type": "application/protobuf"},`)
	t.P(`        )`)
	t.P(`        try:`)
	t.P(`            resp = urllib2.urlopen(req)`)
	t.P(`        except urllib2.HTTPError as err:`)
	t.P(`            raise TwirpException.from_http_err(err)`)
	t.P(``)
	t.P(`        return resp.read()`)
	t.P()

	for _, method := range service.methods {
		methName := methodName(method)
		inputName := methodInputName(method)

		// Be careful not to write code that overwrites the input parameter.
		for _, x := range []string{"self", "_sym_db", "full_method", "body",
			"serialize", "deserialize", "resp_str"} {
			if inputName == x {
				inputName = inputName + "_"
			}
		}

		t.P(`    def `, methName, `(self, `, inputName, `):`)
		if t.gen.commentsAvailable(method.path) {
			t.P(`        """`)
			t.gen.PrintComments(method.path, `        `)
			t.P(`        """`)
			t.P()
		}
		t.P(`        serialize = _sym_db.GetSymbol(`,
			strconv.Quote(strings.TrimPrefix(method.GetInputType(), ".")), `).SerializeToString`)
		t.P(`        deserialize = _sym_db.GetSymbol(`,
			strconv.Quote(strings.TrimPrefix(method.GetOutputType(), ".")), `).FromString`)
		t.P()
		t.P(`        full_method = "/{}/{}".format(self.__service_name, `, strconv.Quote(method.GetName()), `)`)
		t.P(`        body = serialize(`, inputName, `)`)
		t.P(`        resp_str = self.__make_request(body=body, full_method=full_method)`)
		t.P(`        return deserialize(resp_str)`)
		t.P()
	}
}

// Given a type name defined in a .proto, return its name as we will print it.
func (t *twirp) typeName(str string) string {
	return t.gen.TypeName(t.objectNamed(str))
}

// Given a type name defined in a .proto, return its object.
// Also record that we're using it, to guarantee the associated import.
func (t *twirp) objectNamed(name string) Object {
	t.gen.RecordTypeUse(name)
	return t.gen.ObjectNamed(name)
}

func unexported(s string) string { return strings.ToLower(s[:1]) + s[1:] }

func fullServiceName(file *FileDescriptor, service *ServiceDescriptor) string {
	name := CamelCase(service.GetName())
	if pkg := file.GetPackage(); pkg != "" {
		name = pkg + "." + name
	}
	return name
}

func serviceName(service *ServiceDescriptor) string {
	return CamelCase(service.GetName())
}

func clientName(service *ServiceDescriptor) string {
	return CamelCase(service.GetName()) + "Client"
}

func serviceStruct(service *ServiceDescriptor) string {
	return unexported(serviceName(service)) + "Server"
}

func methodName(method *MethodDescriptor) string {
	return SnakeCase(method.GetName())
}
