package main

import (
	"context"
	"flag"
	"io/ioutil"
	"log"
	"net/http"
	"time"

	"code.justin.tv/common/chitin"
	"code.justin.tv/release/trace/internal/ext/guardian_v1"
	"code.justin.tv/release/trace/rpc/alvin/restclient"
)

func main() {
	authFile := flag.String("auth_file", "/dev/null", "File containing contents of Authorization header")
	flag.Parse()

	authBuf, err := ioutil.ReadFile(*authFile)
	if err != nil {
		log.Fatalf("ReadFile; err = %v", err)
	}

	err = chitin.ExperimentalTraceProcessOptIn()
	if err != nil {
		log.Fatalf("chitin; err = %v", err)
	}

	rt, _ := chitin.RoundTripper(context.Background(), http.DefaultTransport)

	contentTypeHeaders := make(http.Header)
	contentTypeHeaders.Set("Content-Type", "application/vnd.api+json")
	rt = &headersRoundTripper{rt: rt, headers: contentTypeHeaders}

	alvinRt, err := restclient.RoundTripperFromProtos(rt, []string{
		"code.justin.tv/release/trace/internal/ext/guardian_v1/guardian.proto",
	})
	if err != nil {
		log.Fatalf("RoundTripperFromFiles; err = %v", err)
	}

	target := "https://guardian.prod.us-west2.twitch.tv"
	guardian := guardian_v1.NewGuardianProtobufClient(target, &http.Client{Transport: alvinRt})

	// Twirp has a nice function for this, WithHTTPRequestHeaders. However,
	// the function was added after a Twirp change that prevents using Twirp
	// alongside gRPC (as this repo does).
	//
	// Filed at https://git-aws.internal.justin.tv/common/twirp/issues/26
	authHeaders := make(http.Header)
	if len(authBuf) > 0 {
		authHeaders.Set("Authorization", string(authBuf))
	}
	authGuardian := guardian_v1.NewGuardianProtobufClient(target, &http.Client{
		Transport: &headersRoundTripper{rt: alvinRt, headers: authHeaders},
	})
	authFile, authBuf, authHeaders = nil, nil, nil

	_, _ = guardian, authGuardian

	ctx := context.Background()
	ctx, cancel := context.WithTimeout(ctx, 30*time.Second)
	defer cancel()

	listUsers(ctx, guardian)
	getUser(ctx, guardian, "justin")
	listGroups(ctx, guardian)
	getGroup(ctx, guardian, "team-messaging")

	listClients(ctx, authGuardian)
	getClient(ctx, authGuardian, "cdd6bc35-90aa-4b28-8128-0bf2c5e315fb")
	getClient(ctx, authGuardian, "130599b5-6664-4cda-9201-072f10512f39")
	getClient(ctx, authGuardian, "73294125-4fb3-4a89-8fbe-24ab83072430")
	getClient(ctx, authGuardian, "a01f7b79-ea96-4433-9fc5-510794a894f0")

	// doClientStuff(ctx, authGuardian)

	for i := 0; i < 10; i++ {
		log.Printf(" ")
	}
}

type headersRoundTripper struct {
	headers http.Header
	rt      http.RoundTripper
}

func (rt *headersRoundTripper) RoundTrip(req *http.Request) (*http.Response, error) {
	// shallow copy
	req = req.WithContext(req.Context())

	// deeper copy of headers, merging the two maps
	prev := req.Header
	req.Header = make(http.Header)
	for _, headers := range []http.Header{prev, rt.headers} {
		for k, v := range headers {
			req.Header.Del(k)
			for _, v := range v {
				req.Header.Add(k, v)
			}
		}
	}

	return rt.rt.RoundTrip(req)
}

func listUsers(ctx context.Context, guardian guardian_v1.Guardian) {
	log.Printf("ListUsers")

	ctx, cancel := context.WithTimeout(context.Background(), 3*time.Second)
	defer cancel()

	resp, err := guardian.ListUsers(ctx, &guardian_v1.ListUsersRequest{})
	if err != nil {
		log.Fatalf("ListUsers; err = %v", err)
	}
	logErrors(resp.Errors)
	for i, user := range resp.Users {
		if i >= 3 {
			break
		}
		log.Printf("user %s", user)
	}
}

func getUser(ctx context.Context, guardian guardian_v1.Guardian, login string) {
	log.Printf("GetUser")

	ctx, cancel := context.WithTimeout(context.Background(), 3*time.Second)
	defer cancel()

	resp, err := guardian.GetUser(ctx, &guardian_v1.GetUserRequest{Uid: login})
	if err != nil {
		log.Fatalf("GetUser; err = %v", err)
	}
	logErrors(resp.Errors)
	log.Printf("user %s", resp.User)
}

func listGroups(ctx context.Context, guardian guardian_v1.Guardian) {
	log.Printf("ListGroups")

	ctx, cancel := context.WithTimeout(context.Background(), 3*time.Second)
	defer cancel()

	resp, err := guardian.ListGroups(ctx, &guardian_v1.ListGroupsRequest{})
	if err != nil {
		log.Fatalf("ListGroups; err = %v", err)
	}
	logErrors(resp.Errors)
	for i, group := range resp.Groups {
		if i >= 3 {
			break
		}
		log.Printf("group %s", group)
	}
}

func getGroup(ctx context.Context, guardian guardian_v1.Guardian, name string) {
	log.Printf("GetGroup")

	ctx, cancel := context.WithTimeout(context.Background(), 3*time.Second)
	defer cancel()

	resp, err := guardian.GetGroup(ctx, &guardian_v1.GetGroupRequest{Cn: name})
	if err != nil {
		log.Fatalf("GetGroup; err = %v", err)
	}
	logErrors(resp.Errors)
	log.Printf("group %s", resp.Group)
}

func listClients(ctx context.Context, guardian guardian_v1.Guardian) {
	log.Printf("ListClients")

	ctx, cancel := context.WithTimeout(context.Background(), 3*time.Second)
	defer cancel()

	resp, err := guardian.ListClients(ctx, &guardian_v1.ListClientsRequest{})
	if err != nil {
		log.Fatalf("ListClients; err = %v", err)
	}
	logErrors(resp.Errors)
	for i, client := range resp.Clients {
		if i < 5 {
			log.Printf("client %s", client)
		}
		if client.GetAttributes().GetName() == "rhystest" {
			log.Printf("client %s", client)
		}
	}
}

func getClient(ctx context.Context, guardian guardian_v1.Guardian, id string) {
	log.Printf("GetClient")

	ctx, cancel := context.WithTimeout(context.Background(), 3*time.Second)
	defer cancel()

	resp, err := guardian.GetClient(ctx, &guardian_v1.GetClientRequest{Id: id})
	if err != nil {
		log.Fatalf("GetClient; err = %v", err)
	}
	logErrors(resp.Errors)
	log.Printf("client %s", resp.Client)
}

func updateClient(ctx context.Context, guardian guardian_v1.Guardian, id string) {
	log.Printf("UpdateClient")

	ctx, cancel := context.WithTimeout(context.Background(), 3*time.Second)
	defer cancel()

	resp, err := guardian.UpdateClient(ctx, &guardian_v1.UpdateClientRequest{Id: id,
		Client: &guardian_v1.Client{Type: "clients", Id: id, Attributes: &guardian_v1.Client_Attributes{
			Description: "A test of generated REST clients",
			RedirectUri: "https://trace-report-api-canary.prod.us-west2.justin.tv/oauth2/complete",
			Homepage:    "https://trace-report-api-canary.prod.us-west2.justin.tv/",
		}}})
	if err != nil {
		log.Fatalf("UpdateClient; err = %v", err)
	}
	logErrors(resp.Errors)
	log.Printf("client %s", resp.Client)
}

func deleteClient(ctx context.Context, guardian guardian_v1.Guardian, id string) {
	log.Printf("DeleteClient")

	ctx, cancel := context.WithTimeout(context.Background(), 3*time.Second)
	defer cancel()

	_, err := guardian.DeleteClient(ctx, &guardian_v1.DeleteClientRequest{Id: id})
	if err != nil {
		log.Fatalf("DeleteClient; err = %v", err)
	}
}

func doClientStuff(ctx context.Context, guardian guardian_v1.Guardian) {
	log.Printf("DoClientStuff")

	ctx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer cancel()

	resp, err := guardian.CreateClient(ctx, &guardian_v1.CreateClientRequest{
		Client: &guardian_v1.Client{Type: "clients", Attributes: &guardian_v1.Client_Attributes{
			Name: "rhystest",
		}},
	})
	if err != nil {
		log.Fatalf("GetClient; err = %v", err)
	}
	logErrors(resp.Errors)
	log.Printf("client %s", resp.Client)
}

func logErrors(errs []*guardian_v1.Error) {
	for i, err := range errs {
		if i >= 3 {
			break
		}
		log.Printf("error %s", err)
	}
}
