package main

import (
	"bytes"
	"context"
	"errors"
	"fmt"
	"html/template"
	"log"
	"net/http"
	"net/url"
	"path"
	"strings"
	"time"

	"code.justin.tv/release/trace/analysis/render"
	"code.justin.tv/release/trace/api/report_v1"
	"code.justin.tv/release/trace/internal/htmlreport"
	"github.com/golang/protobuf/ptypes"
	"github.com/golang/protobuf/ptypes/empty"
	netcontext "golang.org/x/net/context"
	"golang.org/x/net/trace"
)

type browserHandler struct {
	reports  report_v1.Reports
	backfill http.Handler
}

var _ report_v1.ReportBrowser = (*browserHandler)(nil)
var _ http.Handler = (*browserHandler)(nil)

func (br *browserHandler) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	var (
		body *report_v1.HttpBody
		err  error
	)
	switch r.Method {
	case http.MethodGet, http.MethodHead:
		body, err = br.handleGET(r.Context(), r.URL)
	}

	if err != nil {
		if err == notFound {
			http.NotFound(w, r)
			return
		}
		log.Printf("browserHandler err=%q", err)
		http.Error(w, "browserHandler internal error", http.StatusInternalServerError)
		return
	}

	if body != nil {
		w.Header().Set("Content-Type", body.ContentType)
		if r.Method != http.MethodHead {
			fmt.Fprintf(w, "%s", body.Data)
		}
		return
	}

	br.backfill.ServeHTTP(w, r)
	return
}

func (br *browserHandler) handleGET(ctx context.Context, u *url.URL) (*report_v1.HttpBody, error) {
	// This function could be entirely replaced by code that reads
	// google.api.http annotations and either automatically generates the
	// routing instructions or which determines the routing at runtime.
	//
	// The runtime alternative would live at
	// code.justin.tv/release/trace/rpc/alvin/restserver, complementing the
	// restclient package.
	//
	// The code below (and in the rest of this file) might be structured
	// differently if there were no plans for a protobuf-annotation-based
	// auto-router.

	var tr trace.Trace
	nettraceContext := func(name string) context.Context {
		tr = trace.New(name, "")
		tr.SetMaxEvents(30)
		return trace.NewContext(ctx, tr)
	}
	defer func() {
		if tr == nil {
			nettraceContext("missing")
		}
		tr.Finish()
	}()

	switch parts := strings.Split(u.Path, "/"); {
	case parts[0] != "": // odd/invalid URI path
		return nil, nil
	case len(parts) == 2 && parts[1] == "latest":
		// option (google.api.http) = { get: "/latest" };
		ctx = nettraceContext("report_v1.ReportBrowser/GetLatestReport")
		return br.GetLatestReport(ctx, &empty.Empty{})
	case len(parts) == 3 && parts[2] == "report":
		// option (google.api.http) = { get: "/{report_id}/report" };
		ctx = nettraceContext("report_v1.ReportBrowser/ListProgramsInReport")
		return br.ListProgramsInReport(ctx, &report_v1.ListProgramsInReportRequest{
			ReportId: parts[1]},
		)
	case len(parts) >= 5 && parts[2] == "report" && parts[3] == "bin":
		// option (google.api.http) = { get: "/{report_id}/report/bin/{program.name=**}" };
		ctx = nettraceContext("report_v1.ReportBrowser/GetProgramReport")
		return br.GetProgramReport(ctx, &report_v1.GetProgramReportRequest{
			ReportId: parts[1],
			Program:  &report_v1.ProgramRef{Name: strings.Join(parts[4:], "/")},
		})
	case len(parts) == 5 && parts[2] == "report" && parts[3] == "tx":
		// option (google.api.http) = { get: "/{report_id}/report/tx/{transaction_id}" };
		ctx = nettraceContext("report_v1.ReportBrowser/LookupTransaction")
		return br.LookupTransaction(ctx, &report_v1.LookupTransactionRequest{
			ReportId:      parts[1],
			TransactionId: parts[4],
		})
	case len(parts) == 4 && parts[2] == "report" && parts[3] == "tx":
		// option (google.api.http) = { get: "/{report_id}/report/tx" };
		ctx = nettraceContext("report_v1.ReportBrowser/GetTransactionPage")
		return br.GetTransactionPage(ctx, &report_v1.GetTransactionPageRequest{
			ReportId: parts[1],
		})
	case len(parts) >= 4 && parts[2] == "report" && (parts[3] == "css" || parts[3] == "js"):
		// option (google.api.http) = {
		//   get: "/*/report/{asset_name=css/**}"
		//   additional_bindings {
		//     get: "/*/report/{asset_name=js/**}"
		//   }
		// };
		ctx = nettraceContext("report_v1.ReportBrowser/GetAsset")
		return br.GetAsset(ctx, &report_v1.GetAssetRequest{AssetName: strings.Join(parts[3:], "/")})
	}
	return nil, nil
}

type uriGenerator struct {
	prefix string
}

func (p uriGenerator) URIFor(partial string) string { return path.Join(p.prefix, partial) }

func (p uriGenerator) URIForCallRef(ref *report_v1.CallRef) string {
	return path.Join(p.prefix, "tx") + "?" + (url.Values{
		"id": []string{ref.GetTransactionId()},
	}).Encode()
}

func (p uriGenerator) URIForProgramRef(ref *report_v1.ProgramRef) string {
	return path.Join(p.prefix, "bin", ref.GetName())
}

func (br *browserHandler) ListProgramsInReport(ctx netcontext.Context, req *report_v1.ListProgramsInReportRequest) (*report_v1.HttpBody, error) {
	contentType := "text/html; charset=utf-8"

	home := path.Join("/", req.GetReportId(), "report")
	root := path.Join(home, "/")

	resp, err := br.reports.ListProgramsInReport(ctx, req)
	if err != nil {
		return nil, err
	}

	buf, err := htmlreport.RenderProgramIndex(uriGenerator{prefix: root}, resp.GetPrograms())
	if err != nil {
		return nil, err
	}

	return &report_v1.HttpBody{ContentType: contentType, Data: buf}, nil
}

func (br *browserHandler) GetProgramReport(ctx netcontext.Context, req *report_v1.GetProgramReportRequest) (*report_v1.HttpBody, error) {
	contentType := "text/html; charset=utf-8"

	home := path.Join("/", req.GetReportId(), "report")
	root := path.Join(home, "/")

	resp, err := br.reports.GetProgramReport(ctx, req)
	if err != nil {
		return nil, err
	}

	buf, err := htmlreport.RenderProgramReport(uriGenerator{prefix: root}, home, resp.GetContent())
	if err != nil {
		return nil, err
	}

	return &report_v1.HttpBody{ContentType: contentType, Data: buf}, nil
}

func (br *browserHandler) LookupTransaction(ctx netcontext.Context, req *report_v1.LookupTransactionRequest) (*report_v1.HttpBody, error) {
	// TODO
	return &report_v1.HttpBody{
		ContentType: "text/plain; charset=utf-8",
		Data:        []byte(fmt.Sprintf("LookupTransaction unimplemented\n%s\n", req)),
	}, nil
}

func (br *browserHandler) GetTransactionPage(ctx netcontext.Context, req *report_v1.GetTransactionPageRequest) (*report_v1.HttpBody, error) {
	contentType := "text/html; charset=utf-8"

	home := path.Join("/", req.GetReportId(), "report")
	root := path.Join(home, "/")

	buf, err := htmlreport.RenderTransactionBasePage(uriGenerator{prefix: root})
	if err != nil {
		return nil, err
	}

	return &report_v1.HttpBody{ContentType: contentType, Data: buf}, nil
}

func (br *browserHandler) GetLatestReport(ctx netcontext.Context, _ *empty.Empty) (*report_v1.HttpBody, error) {
	contentType := "text/html; charset=utf-8"

	list, err := br.reports.ListReports(ctx, &report_v1.ListReportsRequest{})
	if err != nil {
		return nil, err
	}
	latest := latestReport(list.Report)
	uri := latest.GetId() + "/report"

	var buf bytes.Buffer
	template.Must(template.New("").Parse(`
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta http-equiv="refresh" content="0; url={{.}}" />
  <title>Current Trace Report</title>
</head>
<body>
  <p>The current Trace report is available at <a href="{{.}}">{{.}}</a>.</p>
</body>
</html>
`[1:])).Execute(&buf, uri)

	return &report_v1.HttpBody{ContentType: contentType, Data: buf.Bytes()}, nil
}

func latestReport(reports []*report_v1.Report) *report_v1.Report {
	var latest *report_v1.Report
	var latestTime time.Time
	for _, rep := range reports {
		repTime, err := ptypes.Timestamp(rep.GetModified())
		if err != nil || (!latestTime.IsZero() && repTime.Before(latestTime)) {
			continue
		}
		latest, latestTime = rep, repTime
	}
	return latest
}

func (br *browserHandler) GetAsset(ctx netcontext.Context, req *report_v1.GetAssetRequest) (*report_v1.HttpBody, error) {
	var resp report_v1.HttpBody

	path := req.AssetName
	switch path {
	default:
		return nil, notFound
	case "css/style.css":
		resp.ContentType = "text/css; charset=utf-8"
	case "js/tx.js":
		resp.ContentType = "application/js; charset=utf-8"
	}
	resp.Data = render.MustAsset(path)

	return &resp, nil
}

var notFound = errors.New("not found")
