package htmlreport

import (
	"fmt"
	"math"
	"time"

	"code.justin.tv/release/trace/api/report_v1"
	"github.com/golang/protobuf/ptypes"
)

func seq(start, end, step float64) []float64 {
	reverse := false
	if start > end {
		reverse = true
		start, end = end, start
	}
	out := []float64{}
	for val := start; val <= end; val += step {
		out = append(out, val)
	}
	if reverse {
		reversed := make([]float64, len(out))
		for i, val := range out {
			reversed[len(out)-i-1] = val
		}
		out = reversed
	}
	return out
}

// timePercentile returns the duration in milliseconds of a particular
// percentile.
func timePercentile(d *report_v1.Distribution, p float64) float64 {
	if d == nil || d.GetCount() == 0 {
		return 0
	}

	opts := d.GetBucketOptions().GetExponentialBuckets()
	if opts == nil {
		// TODO: support other bucket types
		return 0
	}

	var val float64
	remaining := int64(math.Floor(float64(d.GetCount())*p + 0.5))
	for i, n := range d.GetBucketCounts() {
		remaining -= n
		if remaining > 0 {
			continue
		}

		if i == 0 {
			val = math.Sqrt(d.Range.Min * opts.Scale)
		} else if i == int(opts.NumFiniteBuckets)+1 {
			val = math.Sqrt(d.Range.Max * opts.Scale * math.Pow(opts.GrowthFactor, float64(opts.NumFiniteBuckets)))
		} else {
			// aim for the geometric midpoint of the bucket
			val = opts.Scale * math.Pow(opts.GrowthFactor, float64(i-1)+0.5)
		}
		break

	}

	if val < d.Range.Min {
		val = d.Range.Min
	}
	if val > d.Range.Max {
		val = d.Range.Max
	}
	return val / time.Millisecond.Seconds()
}

func subtreeCompactDescription(sum *report_v1.SubtreeSummary) string {
	dur, err := ptypes.Duration(sum.GetCall().GetServerDuration())
	if err != nil {
		dur = 0
	}
	ms := dur.Seconds() / time.Millisecond.Seconds()
	return fmt.Sprintf("dur:%.2fms  depth:%d  size:%d",
		ms, sum.GetSubtreeDepth(), sum.GetSubtreeSize())
}
