package httproute

import (
	"net/url"
	"reflect"
	"testing"
)

func TestTemplateIsMatch(t *testing.T) {
	for _, tt := range []struct {
		tmpl  string
		path  string
		match bool
	}{

		{tmpl: "/he:llo", path: "/he:llo", match: true},
		{tmpl: "/he:llo", path: "/he:y", match: false},
		{tmpl: "/hello", path: "/hello", match: true},
		{tmpl: "/hello", path: "/world", match: false},
		{tmpl: "/hello", path: "/hello:", match: true},
		{tmpl: "/*", path: "/hello", match: true},
		{tmpl: "/**", path: "/hello", match: true},
		{tmpl: "/**", path: "/", match: true},
		{tmpl: "/**", path: "/hello/world", match: true},

		{tmpl: "/hello", path: "hello", match: false},
		{tmpl: "/hello", path: "", match: false},
		{tmpl: "/*", path: "hello", match: false},
		{tmpl: "/*", path: "", match: false},
		{tmpl: "/**", path: "hello", match: false},
		{tmpl: "/**", path: "", match: false},

		{tmpl: "/hello", path: "/hello/", match: true},
		{tmpl: "/hello", path: "/hello/world", match: false},
		{tmpl: "/hello", path: "/hello/world/", match: false},
		{tmpl: "/*", path: "/hello/", match: true},
		{tmpl: "/*", path: "/hello/world", match: false},
		{tmpl: "/*", path: "/hello/world/", match: false},

		{tmpl: "/{report_id}/report", path: "/2016-12-20-19-55-06.749755/report", match: true},
		{tmpl: "/{report_id}/report/bin/{program.name=**}",
			path: "/2016-12-20-19-55-06.749755/report/bin/code.justin.tv/video/usher/api/usher", match: true},
		{tmpl: "/{report_id}/report/tx/{transaction_id}",
			path: "/2016-12-20-19-55-06.749755/report/tx/79a35068bec54ad7b888c5c1df1821ff", match: true},
		{tmpl: "/{report_id}/report/tx/{transaction_id}",
			path: "/2016-12-20-19-55-06.749755/report/tx", match: false},
		{tmpl: "/*/report/tx",
			path: "/2016-12-20-19-55-06.749755/report/tx/79a35068bec54ad7b888c5c1df1821ff", match: false},
		{tmpl: "/*/report/tx",
			path: "/2016-12-20-19-55-06.749755/report/tx", match: true},

		{tmpl: "/v2/{table_name=projects/*/instances/*/tables/*}:mutateRow",
			path: "/v2/projects/foo/instances/bar/tables/baz:mutateRow", match: true},
		{tmpl: "/v2/{table_name=projects/*/instances/*/tables/*}:mutateRow",
			path: "/v2/projects/foo/instances/bar/tables/baz:derp", match: false},
		{tmpl: "/v2/{table_name=projects/*/instances/*/tables/*}:mutateRow",
			path: "/v2/projects/foo/instances/tables/baz:mutateRow", match: false},
	} {
		t.Run("", func(t *testing.T) {
			tmpl, err := ParseTemplate(tt.tmpl)
			if err != nil {
				t.Fatalf("parseTemplate(%q); err = %v", tt.tmpl, err)
			}
			if have, want := tmpl.isMatch(tt.path), tt.match; have != want {
				t.Errorf("parseTemplate(%q).isMatch(%q); %t != %t", tt.tmpl, tt.path, have, want)
				t.Logf("template: %#v", tmpl)
			}
		})
	}
}

func TestPathMatcher(t *testing.T) {
	t.Run("invalid template", func(t *testing.T) {
		_, err := NewPathMatcher([]string{""})
		if err == nil {
			t.Fatalf("NewPathMatcher([]string{\"\"}); err = nil")
		}
	})

	t.Run("bad roundtrip", func(t *testing.T) {
		_, err := newPathMatcher([]string{"/"}, func(path string) (*Template, error) {
			return &Template{segments: tmplSegments{{literal: "nope"}}}, nil
		})
		if err == nil {
			t.Fatalf("newPathMatcher([]string{\"/\"}, nonRoundtripTemplateGenerator); err = nil")
		}
	})

	t.Run("match", func(t *testing.T) {
		matcher, err := NewPathMatcher([]string{
			"/he:llo",
			"/he:y",
			"/*:world",

			"/one/two",
			"/*/two",

			"/{report_id}/report",
			"/{report_id}/report/bin/{program.name=**}",
			"/{report_id}/report/tx/{transaction_id}",
			"/*/report/tx",
			"/latest",
		})
		if err != nil {
			t.Fatalf("NewPathMatcher; err = %v", err)
		}
		for _, tt := range []struct {
			path   string
			tmpl   string
			values url.Values
		}{
			{tmpl: "/he:llo", path: "/he:llo", values: url.Values{}},
			{tmpl: "/he:y", path: "/he:y", values: url.Values{}},
			{tmpl: "/*:world", path: "/he:world", values: url.Values{}},
			{tmpl: "/*:world", path: "/hello:world", values: url.Values{}},

			{tmpl: "/one/two", path: "/one/two", values: url.Values{}},
			{tmpl: "/*/two", path: "/route/two", values: url.Values{}},

			{
				tmpl:   "/latest",
				path:   "/latest",
				values: url.Values{},
			},
			{
				tmpl: "/{report_id}/report",
				path: "/2016-12-20-19-55-06.749755/report",
				values: url.Values{
					"report_id": []string{"2016-12-20-19-55-06.749755"},
				},
			},
			{
				tmpl: "/{report_id}/report/bin/{program.name=**}",
				path: "/2016-12-20-19-55-06.749755/report/bin/code.justin.tv/video/usher/api/usher",
				values: url.Values{
					"report_id":    []string{"2016-12-20-19-55-06.749755"},
					"program.name": []string{"code.justin.tv/video/usher/api/usher"},
				},
			},
			{
				tmpl: "/{report_id}/report/bin/{program.name=**}",
				path: "/2016-12-20-19-55-06.749755/report/bin/",
				values: url.Values{
					"report_id":    []string{"2016-12-20-19-55-06.749755"},
					"program.name": []string{""},
				},
			},
			{
				tmpl: "/{report_id}/report/bin/{program.name=**}",
				path: "/2016-12-20-19-55-06.749755/report/bin",
				values: url.Values{
					"report_id":    []string{"2016-12-20-19-55-06.749755"},
					"program.name": []string{""},
				},
			},
			{
				tmpl:   "/*/report/tx",
				path:   "/2016-12-20-19-55-06.749755/report/tx",
				values: url.Values{},
			},
			{
				tmpl: "/{report_id}/report/tx/{transaction_id}",
				path: "/2016-12-20-19-55-06.749755/report/tx/79a35068bec54ad7b888c5c1df1821ff",
				values: url.Values{
					"report_id":      []string{"2016-12-20-19-55-06.749755"},
					"transaction_id": []string{"79a35068bec54ad7b888c5c1df1821ff"},
				},
			},
		} {
			template, values := matcher.Match(tt.path)
			if have, want := template, tt.tmpl; have != want {
				t.Fatalf("Match(%q); %q != %q", tt.path, have, want)
			}
			if have, want := values, tt.values; !reflect.DeepEqual(have, want) {
				t.Errorf("Match(%q);\n%#v\n!=\n%#v", tt.path, have, want)
			}
		}
	})
}
