package restclient

import (
	"bytes"
	"compress/gzip"
	"context"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"net/http"
	"net/http/httptest"
	"net/url"
	"reflect"
	"strings"
	"testing"
	"time"

	"code.justin.tv/common/twirp"
	"code.justin.tv/release/trace/rpc/alvin/internal/httpproto"
	"code.justin.tv/release/trace/rpc/alvin/internal/httproute"
	"code.justin.tv/release/trace/rpc/alvin/internal/testproto"
	"github.com/golang/protobuf/ptypes/empty"
	structpb "github.com/golang/protobuf/ptypes/struct"
	"github.com/pkg/errors"
)

func TestRoundTripper(t *testing.T) {
	// Until there's an implementation of the RoundTripper, we'll use this
	// test to get the package to 100% line coverage.
	defer func() {
		recover()
	}()
	(&roundTripper{}).RoundTrip(nil)
	t.Fatalf("expected panic")
}

func TestTwirpRPCMethod(t *testing.T) {
	for _, tt := range []struct {
		in, out string
	}{
		{"/", ""},
		{"/hello", ""},
		{"/some/path", ""},
		{"/some/path/or/another", ""},

		{"/v1/Service/Method", ""},
		{"/v2/fully.qualified.Service/Method", "/fully.qualified.Service/Method"},
		{"/twirp/fully.qualified.Service/Method", "/fully.qualified.Service/Method"},

		// We could validate these more and reject them early, but we'll soon
		// find they're missing from the route map. Early rejection would be
		// beneficial in the name of "full recognition before processing".
		{"/v1/bogus/path/made/up/for/fun", ""},
		{"/v2/bogus/path/made/up/for/fun", "/bogus/path/made/up/for/fun"},
		{"/twirp/bogus/path/made/up/for/fun", "/bogus/path/made/up/for/fun"},
		{"/v1/short", ""},
		{"/v2/short", "/short"},
		{"/twirp/short", "/short"},
	} {
		if have, want := twirpRPCMethod(tt.in), tt.out; have != want {
			t.Errorf("twirpRPCMethod(%q); %q != %q", tt.in, have, want)
		}
	}
}

func TestNewRouteSpec(t *testing.T) {
	t.Run("no mappings", func(t *testing.T) {
		_, err := newRouteSpec(&httpproto.MethodSpec{})
		checkError(t, "newRouteSpec; ", err, "no HTTP mappings")
	})

	t.Run("allow nested fields", func(t *testing.T) {
		_, err := newRouteSpec(&httpproto.MethodSpec{
			FullName:   "/code_justin_tv.release.trace.rpc.alvin.internal.testproto.MyService3/MethodTwo",
			InputType:  ".code_justin_tv.release.trace.rpc.alvin.internal.testproto.RequestA3",
			OutputType: ".code_justin_tv.release.trace.rpc.alvin.internal.testproto.ResponseA3",
			Routes: []*httpproto.RouteSpec{
				{
					HTTPMethod:  "POST",
					PathPattern: "/route/to/{deeply.nested.field=*}",
					BodyField:   "deeply.nested",
				},
			},
		})
		checkError(t, "newRouteSpec; ", err, "")
	})

	t.Run("invalid template", func(t *testing.T) {
		_, err := newRouteSpec(&httpproto.MethodSpec{
			FullName:   "/code_justin_tv.release.trace.rpc.alvin.internal.testproto.MyService3/MethodTwo",
			InputType:  ".code_justin_tv.release.trace.rpc.alvin.internal.testproto.RequestA3",
			OutputType: ".code_justin_tv.release.trace.rpc.alvin.internal.testproto.ResponseA3",
			Routes: []*httpproto.RouteSpec{
				{
					HTTPMethod:  "POST",
					PathPattern: "{", // <- the bad line
					BodyField:   "deeply.nested",
				},
			},
		})
		checkError(t, "newRouteSpec; ", err, "invalid template")
	})

	t.Run("invalid input type", func(t *testing.T) {
		_, err := newRouteSpec(&httpproto.MethodSpec{
			FullName:   "/code_justin_tv.release.trace.rpc.alvin.internal.testproto.MyService3/MethodTwo",
			InputType:  "int", // <- the bad line
			OutputType: ".code_justin_tv.release.trace.rpc.alvin.internal.testproto.ResponseA3",
			Routes: []*httpproto.RouteSpec{
				{
					HTTPMethod:  "POST",
					PathPattern: "/route/to/{deeply.nested.field=*}",
					BodyField:   "deeply.nested",
				},
			},
		})
		checkError(t, "newRouteSpec; ", err, "unknown protobuf type \"int\" ")
	})

	t.Run("invalid output type", func(t *testing.T) {
		_, err := newRouteSpec(&httpproto.MethodSpec{
			FullName:   "/code_justin_tv.release.trace.rpc.alvin.internal.testproto.MyService3/MethodTwo",
			InputType:  ".code_justin_tv.release.trace.rpc.alvin.internal.testproto.RequestA3",
			OutputType: "bool", // <- the bad line
			Routes: []*httpproto.RouteSpec{
				{
					HTTPMethod:  "POST",
					PathPattern: "/route/to/{deeply.nested.field=*}",
					BodyField:   "deeply.nested",
				},
			},
		})
		checkError(t, "newRouteSpec; ", err, "unknown protobuf type \"bool\" ")
	})
}

func TestNewPointer(t *testing.T) {
	t.Run("int", func(t *testing.T) {
		p := newPointer(reflect.TypeOf(0))
		if p == nil {
			t.Fatalf("newPointer; p = nil")
		}
		ip, ok := p.(*int)
		if !ok {
			t.Fatalf("newPointer; type = %T", p)
		}
		if ip == nil {
			t.Fatalf("newPointer; ip = nil")
		}
		if *ip != 0 {
			t.Fatalf("newPointer; *ip != 0")
		}
	})

	t.Run("*int", func(t *testing.T) {
		p := newPointer(reflect.TypeOf(new(int)))
		if p == nil {
			t.Fatalf("newPointer; p = nil")
		}
		ip, ok := p.(*int)
		if !ok {
			t.Fatalf("newPointer; type = %T", p)
		}
		if ip == nil {
			t.Fatalf("newPointer; ip = nil")
		}
		if *ip != 0 {
			t.Fatalf("newPointer; *ip != 0")
		}
	})
}

func TestReadInputBody(t *testing.T) {
	t.Run("bad input type", func(t *testing.T) {
		msg := new(int)
		err := readBody(msg, "application/json", ioutil.NopCloser(strings.NewReader(`{}`)))
		checkError(t, "readInputBody; ", err, "non-protobuf message type")
	})

	t.Run("bad content type", func(t *testing.T) {
		msg := new(testproto.RequestA3)
		err := readBody(msg, "text/plain", ioutil.NopCloser(strings.NewReader("hello world")))
		checkError(t, "readInputBody; ", err, "unknown content type")
	})

	t.Run("bad json", func(t *testing.T) {
		msg := new(testproto.RequestA3)
		err := readBody(msg, "application/json", ioutil.NopCloser(strings.NewReader(`{`)))
		checkError(t, "readInputBody; ", err, "unexpected EOF")
		buf := err.(*bodyErr).body
		if have, want := string(buf), "{"; have != want {
			t.Errorf("invalid response body not returned to caller; %q != %q", have, want)
		}
	})

	t.Run("json contents", func(t *testing.T) {
		msg := new(testproto.RequestA3)
		err := readBody(msg, "application/json", ioutil.NopCloser(strings.NewReader(`{"deeply":{"buried":["treasure"]}}`)))
		checkError(t, "readInputBody; ", err, "")
		abortIfFailed(t)
		if have, want := msg.Deeply.Buried[0], "treasure"; have != want {
			t.Errorf("msg.Deeply.Buried[0]; %q != %q", have, want)
		}
	})

	t.Run("empty protobuf", func(t *testing.T) {
		msg := new(testproto.RequestA3)
		err := readBody(msg, "application/protobuf", ioutil.NopCloser(strings.NewReader("")))
		checkError(t, "readInputBody; ", err, "")
	})
	t.Run("bad protobuf", func(t *testing.T) {
		msg := new(testproto.RequestA3)
		err := readBody(msg, "application/protobuf", ioutil.NopCloser(strings.NewReader("\x0F")))
		checkError(t, "readInputBody; ", err, "bad wiretype")
		buf := err.(*bodyErr).body
		if have, want := string(buf), "\x0F"; have != want {
			t.Errorf("invalid response body not returned to caller; %q != %q", have, want)
		}
	})
	t.Run("protobuf io.Reader error", func(t *testing.T) {
		// make a body read fail
		var buf bytes.Buffer
		w := gzip.NewWriter(&buf)
		fmt.Fprintf(w, "hello world")
		w.Close()
		r, _ := gzip.NewReader(bytes.NewReader(buf.Bytes()[:buf.Len()/2]))

		msg := new(testproto.RequestA3)
		err := readBody(msg, "application/protobuf", ioutil.NopCloser(r))
		b := err.(*bodyErr).body
		checkError(t, "readInputBody; ", err, "unexpected EOF")
		if have, want := string(b), "hello world"; !strings.HasPrefix(want, have) {
			t.Errorf("invalid response body not returned to caller; %q is not a prefix of %q", have, want)
		}

		if have, want := errors.Cause(err), io.ErrUnexpectedEOF; have != want {
			t.Errorf("error Cause; %q != %q", have, want)
		}
	})

	t.Run("non-`Empty` empty message", func(t *testing.T) {
		msg := new(testproto.ResponseA3)
		err := readBody(msg, "application/json", ioutil.NopCloser(strings.NewReader("")))
		buf := err.(*bodyErr).body
		checkError(t, "readInputBody; ", err, "EOF")
		if have, want := string(buf), ""; have != want {
			t.Errorf("invalid response body not returned to caller; %q != %q", have, want)
		}
	})
	t.Run("google.protobuf.Empty", func(t *testing.T) {
		msg := new(empty.Empty)
		err := readBody(msg, "application/json", ioutil.NopCloser(strings.NewReader("")))
		if err != nil {
			t.Fatalf("readBody; err = %v", err)
		}
	})
}

func TestMarshalBody(t *testing.T) {
	testcase := func(msg interface{}, contentType string, want []byte) func(t *testing.T) {
		return func(t *testing.T) {
			buf, err := marshalBody(msg, contentType)
			if err != nil {
				t.Fatalf("marshalBody(%q, %q); err = %v", msg, contentType, err)
			}
			if have, want := string(buf), string(want); have != want {
				t.Errorf("marshalBody(%q, %q); %q != %q", msg, contentType, have, want)
			}
		}
	}

	errcase := func(msg interface{}, contentType string, errString string) func(t *testing.T) {
		return func(t *testing.T) {
			_, err := marshalBody(msg, contentType)
			checkError(t,
				fmt.Sprintf("marshalBody(%q, %q); ", msg, contentType),
				err, errString)
		}
	}

	t.Run("json encoding", testcase(
		&testproto.RequestA3_SubTwo{Field: 66},
		"application/json", []byte(`{"field":"66"}`)))

	t.Run("protobuf encoding", testcase(
		&testproto.RequestA3_SubTwo{Field: 66},
		"application/protobuf", []byte{1<<3 | 0, 66}))

	t.Run("non-protobuf input message", errcase(
		new(int), "application/json", "non-protobuf message"))

	t.Run("invalid json", errcase(
		&badJSONHolder{},
		"application/json", "error calling MarshalJSON"))

	t.Run("invalid encoding", errcase(
		&testproto.RequestA3_SubTwo{Field: 66},
		"text/plain", "unknown content type"))
}

func TestEncodeURIPath(t *testing.T) {
	t.Run("message field", func(t *testing.T) {
		input := &testproto.BitOfEverything3{
			SingleMessage: &testproto.BitOfEverything3_SmallMessage{Value: 22},
		}
		tmpl, err := httproute.ParseTemplate("/{single_message}")
		if err != nil {
			t.Fatalf("ParseTemplate; err = %v", err)
		}
		_, err = encodeURIPath(input, tmpl)
		checkError(t, "encodeURIPath; ", err, "non-primitive field")
	})

	t.Run("non-match", func(t *testing.T) {
		input := &testproto.BitOfEverything3{
			SingleMessage: &testproto.BitOfEverything3_SmallMessage{Value: 22},
		}
		tmpl, err := httproute.ParseTemplate("/{single_message.value=foo/*}")
		if err != nil {
			t.Fatalf("ParseTemplate; err = %v", err)
		}
		_, err = encodeURIPath(input, tmpl)
		checkError(t, "encodeURIPath; ", err, "does not match template")
	})
}

func TestEncodeBody(t *testing.T) {
	testcase := func(input interface{}, bodyField string, body string, values string) func(t *testing.T) {
		return func(t *testing.T) {
			buf, vals, err := encodeBody(input, bodyField)
			if err != nil {
				t.Fatalf("encodeBody; err = %v", err)
			}
			if have, want := vals.Encode(), values; have != want {
				t.Errorf("encodeBody; query string %q != %q", have, want)
			}
			if have, want := string(buf), body; have != want {
				t.Errorf("encodeBody; body %q != %q", have, want)
			}
		}
	}

	errcase := func(input interface{}, bodyField string, errString string) func(t *testing.T) {
		return func(t *testing.T) {
			_, _, err := encodeBody(input, bodyField)
			checkError(t,
				fmt.Sprintf("encodeBody(%q, %q); ", input, bodyField),
				err, errString)
		}
	}

	t.Run("star body", testcase(&testproto.BitOfEverything3{
		SingleMessage: &testproto.BitOfEverything3_SmallMessage{Value: 22},
	}, "*", `{"singleMessage":{"value":22}}`, ""))

	t.Run("empty message field", testcase(&testproto.BitOfEverything3{},
		"single_message", `null`, ""))

	t.Run("invalid star json body", errcase(&badJSONHolder{}, "*", "error calling MarshalJSON"))

	t.Run("invalid json body", errcase(&testproto.BitOfEverything3{
		Value: &structpb.Value{},
	}, "value", "nil Value"))

	t.Run("invalid leaf field", errcase(&testproto.BitOfEverything3{
		SingleMessage: &testproto.BitOfEverything3_SmallMessage{Value: 22},
		Value:         &structpb.Value{},
	}, "single_message", "cannot format well-known type"))

	t.Run("missing field", errcase(&testproto.BitOfEverything3{},
		"missing_field", "not found"))
}

func TestInvokeREST(t *testing.T) {
	// This function is mostly covered by other tests. There are only a few
	// details to check here.

	errcase := func(httpMethod string, body []byte, input, output interface{}, errString string) func(t *testing.T) {
		return func(t *testing.T) {
			srv := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusOK)
				fmt.Fprintf(w, "%s", body)
			}))
			defer srv.Close()

			target, err := url.Parse(srv.URL)
			if err != nil {
				t.Fatalf("url.Parse; err = %v", err)
			}

			rt := &roundTripper{base: http.DefaultTransport}

			ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
			defer cancel()

			route := &routeSpec{
				bodyField:  "single_message",
				httpMethod: httpMethod,
			}
			route.uriTemplate, err = httproute.ParseTemplate("/**")
			if err != nil {
				t.Fatalf("httproute.ParseTemplate; err = %v", err)
			}

			_, err = rt.invokeREST(ctx, route, input, output, target, nil)
			checkError(t, "invokeREST; ", err, errString)
		}
	}

	t.Run("mismatched input body", errcase("POST", nil,
		&testproto.BitOfEverything3_SmallMessage{Value: 22},
		&testproto.BitOfEverything3{},
		" field "))

	t.Run("http request creation error", errcase("SPACE JUMP", nil,
		&testproto.BitOfEverything3{},
		&testproto.BitOfEverything3{},
		"invalid method"))

	t.Run("invalid output type", errcase("POST", nil,
		&testproto.BitOfEverything3{},
		new(int),
		"non-protobuf message type"))
}

func TestCreateResponse(t *testing.T) {
	// This function is mostly covered by other tests. There are only a few
	// details to check here.

	t.Run("copy header and trailer", func(t *testing.T) {
		resp, err := createResponse(
			&testproto.BitOfEverything3{},
			"application/protobuf",
			&http.Response{
				Header:  http.Header{"header-key": []string{"value1"}},
				Trailer: http.Header{"trailer-key": []string{"value2"}},
			})
		if err != nil {
			t.Fatalf("createResponse; err = %v", err)
		}
		if have, want := resp.Header["Header-Key"], []string{"value1"}; !reflect.DeepEqual(have, want) {
			t.Errorf("createResponse; header %q != %q", have, want)
			t.Logf("header map: %#v", resp.Header)
		}
		if have, want := resp.Trailer["Trailer-Key"], []string{"value2"}; !reflect.DeepEqual(have, want) {
			t.Errorf("createResponse; trailer %q != %q", have, want)
			t.Logf("trailer map: %#v", resp.Trailer)
		}
	})

	t.Run("marshal error", func(t *testing.T) {
		_, err := createResponse(
			&testproto.BitOfEverything3{
				Value: &structpb.Value{},
			},
			"application/json",
			&http.Response{StatusCode: 200})
		checkError(t, "createResponse; ", err, "nil Value")
	})
}

func TestRoundTripperFromFiles(t *testing.T) {
	t.Run("no files", func(t *testing.T) {
		_, err := RoundTripperFromProtos(nil, []string{})
		checkError(t, "RoundTripperFromFiles; ", err, "")
	})

	t.Run("missing file", func(t *testing.T) {
		_, err := RoundTripperFromProtos(nil, []string{"missing.file"})
		checkError(t, "RoundTripperFromFiles; ", err, "not found")
	})

	t.Run("duplicate file", func(t *testing.T) {
		_, err := RoundTripperFromProtos(nil, []string{
			"code.justin.tv/release/trace/rpc/alvin/internal/testproto/test3.proto",
			"code.justin.tv/release/trace/rpc/alvin/internal/testproto/test3.proto",
		})
		checkError(t, "RoundTripperFromFiles; ", err, "duplicate entry")
	})

	t.Run("bad route", func(t *testing.T) {
		_, err := RoundTripperFromProtos(nil, []string{
			"code.justin.tv/release/trace/rpc/alvin/internal/testproto/bad_route.proto",
		})
		checkError(t, "RoundTripperFromFiles; ", err, "invalid template")
	})

	t.Run("load two files", func(t *testing.T) {
		rt, err := RoundTripperFromProtos(nil, []string{
			"code.justin.tv/release/trace/rpc/alvin/internal/testproto/test2.proto",
			"code.justin.tv/release/trace/rpc/alvin/internal/testproto/test3.proto",
		})
		checkError(t, "RoundTripperFromFiles; ", err, "")
		if rt == nil {
			t.Fatalf("RoundTripperFromFiles; rt = nil")
		}
		if have, want := rt.(*roundTripper).base, http.DefaultTransport; have != want {
			t.Errorf("RoundTripperFromFiles.base; %#v != %#v", have, want)
		}
		if have, want := len(rt.(*roundTripper).routes), 1; have < want {
			t.Errorf("RoundTripperFromFiles.routes.len; %d < %d", have, want)
		}
	})
}

func TestRoundTrip(t *testing.T) {
	closeBody := func(resp *http.Response) {
		if resp != nil {
			resp.Body.Close()
		}
	}

	t.Run("", func(t *testing.T) {
		rt, err := RoundTripperFromProtos(nil, []string{
			"code.justin.tv/release/trace/rpc/alvin/internal/testproto/test2.proto",
			"code.justin.tv/release/trace/rpc/alvin/internal/testproto/test3.proto",
		})
		checkError(t, "RoundTripperFromFiles; ", err, "")

		client := &http.Client{Transport: rt}

		t.Run("GET", func(t *testing.T) {
			resp, err := client.Get("/")
			defer closeBody(resp)
			checkError(t, "RoundTripperFromFiles; ", err, "unexpected method")
		})

		t.Run("POST non-twirp", func(t *testing.T) {
			resp, err := client.Post("/", "", nil)
			defer closeBody(resp)
			checkError(t, "RoundTripperFromFiles; ", err, "no route for uri path")
		})

		t.Run("POST missing", func(t *testing.T) {
			resp, err := client.Post("/v2/fully.qualified.Service/Method", "", nil)
			defer closeBody(resp)
			checkError(t, "RoundTripperFromFiles; ", err, "no route for method")
		})

		t.Run("POST present", func(t *testing.T) {
			resp, err := client.Post("/v2/code_justin_tv.release.trace.rpc.alvin.internal.testproto.MyService3/MethodTwo", "", nil)
			defer closeBody(resp)
			checkError(t, "RoundTripperFromFiles; ", err, "field")
		})

		t.Run("POST bad body", func(t *testing.T) {
			resp, err := client.Post("/v2/code_justin_tv.release.trace.rpc.alvin.internal.testproto.MyService3/MethodTwo",
				"application/json", strings.NewReader(`{`))
			defer closeBody(resp)
			checkError(t, "RoundTripperFromFiles; ", err, "unexpected EOF")
		})

		t.Run("POST body", func(t *testing.T) {
			resp, err := client.Post("/v2/code_justin_tv.release.trace.rpc.alvin.internal.testproto.MyService3/MethodTwo",
				"application/json", strings.NewReader(`{"deeply":{"buried":["treasure"]}}`))
			defer closeBody(resp)
			checkError(t, "RoundTripperFromFiles; ", err, "field")
		})

		t.Run("POST body", func(t *testing.T) {
			resp, err := client.Post("http://localhost:80000/v2/code_justin_tv.release.trace.rpc.alvin.internal.testproto.MyService3/MethodTwo",
				"application/json", strings.NewReader(`{"deeply":{"nested":{"field":66}}}`))
			defer closeBody(resp)
			checkError(t, "RoundTripperFromFiles; ", err, "invalid port")
		})

		t.Run("POST success", func(t *testing.T) {
			srv := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if have, want := r.Method, "POST"; have != want {
					t.Errorf("r.Method; %q != %q", have, want)
				}
				if have, want := r.URL.String(), "/route/to/66?deeply.buried=treasure&deeply.buried=map"; have != want {
					t.Errorf("r.URL.String; %q != %q", have, want)
				}
				buf, err := ioutil.ReadAll(r.Body)
				checkError(t, "RoundTripperFromFiles; Body; ", err, "")
				if have, want := string(buf), `{"easter_egg":["red","green"]}`; have != want {
					t.Errorf("Body; %q != %q", have, want)
				}
				fmt.Fprint(w, `{"success": true, "missing_field": 0}`)
			}))
			defer srv.Close()

			resp, err := client.Post(srv.URL+"/v2/code_justin_tv.release.trace.rpc.alvin.internal.testproto.MyService3/MethodTwo",
				"application/json", strings.NewReader(`{"deeply":{"nested":{"field":66,"easter_egg":["red","green"]},"buried":["treasure","map"]}}`))
			defer closeBody(resp)
			checkError(t, "RoundTripperFromFiles; ", err, "")
			abortIfFailed(t)
			defer closeBody(resp)

			buf, err := ioutil.ReadAll(resp.Body)
			checkError(t, "RoundTripperFromFiles; ", err, "")
			if have, want := string(buf), `{"success":true}`; have != want {
				t.Errorf("resp.Body; %q != %q", have, want)
			}
		})
	})
}

func TestServerErrors(t *testing.T) {
	call := func(ctx context.Context, t *testing.T,
		req *testproto.RequestA3, handler http.HandlerFunc) (*testproto.ResponseA3, error) {

		rt, err := RoundTripperFromProtos(nil, []string{
			"code.justin.tv/release/trace/rpc/alvin/internal/testproto/test3.proto",
		})
		if err != nil {
			return nil, err
		}
		srv := httptest.NewServer(handler)
		srv.Config.ErrorLog = log.New(&testLogWriter{TB: t}, "", 0)
		defer srv.Close()
		client := testproto.NewMyService3ProtobufClient(srv.URL, &http.Client{Transport: rt})
		return client.MethodOne(ctx, req)
	}

	errCase := func(code twirp.ErrorCode, handler http.HandlerFunc) func(t *testing.T) {
		return func(t *testing.T) {
			ctx := context.Background()
			req := &testproto.RequestA3{Deeply: &testproto.RequestA3_SubOne{Nested: &testproto.RequestA3_SubTwo{Field: 1}}}

			_, err := call(ctx, t, req, handler)
			if err == nil {
				t.Fatalf("expected to get error")
			}
			t.Logf("err = %v", err)

			twerr, ok := err.(twirp.Error)
			if !ok {
				t.Fatalf("error was not a twirp.Error: type %T", err)
			}
			if have, want := twerr.Code(), code; have != want {
				t.Errorf("twerr.Code; %s != %s", have, want)
			}
		}
	}

	okCase := func(code twirp.ErrorCode, handler http.HandlerFunc) func(t *testing.T) {
		return func(t *testing.T) {
			ctx := context.Background()
			req := &testproto.RequestA3{Deeply: &testproto.RequestA3_SubOne{Nested: &testproto.RequestA3_SubTwo{Field: 1}}}

			_, err := call(ctx, t, req, handler)
			if err != nil {
				t.Fatalf("call; err = %v", err)
			}
		}
	}

	t.Run("known statuses", func(t *testing.T) {
		t.Run("400", errCase(twirp.InvalidArgument, func(w http.ResponseWriter, r *http.Request) { w.WriteHeader(400) }))
		t.Run("401", errCase(twirp.Unauthenticated, func(w http.ResponseWriter, r *http.Request) { w.WriteHeader(401) }))
		t.Run("403", errCase(twirp.PermissionDenied, func(w http.ResponseWriter, r *http.Request) { w.WriteHeader(403) }))
		t.Run("404", errCase(twirp.NotFound, func(w http.ResponseWriter, r *http.Request) { w.WriteHeader(404) }))
		t.Run("not found", errCase(twirp.NotFound, func(w http.ResponseWriter, r *http.Request) { http.NotFound(w, r) }))
		t.Run("408", errCase(twirp.DeadlineExceeded, func(w http.ResponseWriter, r *http.Request) { w.WriteHeader(408) }))
		t.Run("409", errCase(twirp.AlreadyExists, func(w http.ResponseWriter, r *http.Request) { w.WriteHeader(409) }))
		t.Run("412", errCase(twirp.FailedPrecondition, func(w http.ResponseWriter, r *http.Request) { w.WriteHeader(412) }))

		t.Run("500", errCase(twirp.Internal, func(w http.ResponseWriter, r *http.Request) { w.WriteHeader(500) }))
		t.Run("501", errCase(twirp.Unimplemented, func(w http.ResponseWriter, r *http.Request) { w.WriteHeader(501) }))
		t.Run("503", errCase(twirp.Unavailable, func(w http.ResponseWriter, r *http.Request) { w.WriteHeader(503) }))
	})

	t.Run("misc statuses", func(t *testing.T) {
		t.Run("422", errCase(twirp.Internal, func(w http.ResponseWriter, r *http.Request) { w.WriteHeader(422) }))
		t.Run("429", errCase(twirp.Internal, func(w http.ResponseWriter, r *http.Request) { w.WriteHeader(429) }))
		t.Run("502", errCase(twirp.Internal, func(w http.ResponseWriter, r *http.Request) { w.WriteHeader(502) }))
		t.Run("504", errCase(twirp.Internal, func(w http.ResponseWriter, r *http.Request) { w.WriteHeader(504) }))
	})

	t.Run("misc behavior", func(t *testing.T) {
		t.Run("EOF", errCase(twirp.Internal, func(w http.ResponseWriter, r *http.Request) {
			panic("oops")
		}))
		t.Run("redirect", errCase(twirp.Internal, func(w http.ResponseWriter, r *http.Request) {
			http.Redirect(w, r, "/", http.StatusTemporaryRedirect)
		}))
		t.Run("empty 200", errCase(twirp.Internal, func(w http.ResponseWriter, r *http.Request) {
			w.WriteHeader(http.StatusOK)
		}))
		t.Run("text OK", errCase(twirp.Internal, func(w http.ResponseWriter, r *http.Request) {
			fmt.Fprintf(w, "OK")
		}))

		t.Run("OK object", okCase(twirp.DataLoss, func(w http.ResponseWriter, r *http.Request) {
			fmt.Fprintf(w, "{}")
		}))
		t.Run("201 object", okCase(twirp.DataLoss, func(w http.ResponseWriter, r *http.Request) {
			w.WriteHeader(201)
			fmt.Fprintf(w, "{}")
		}))
	})

	t.Run("unmarshal prefix", func(t *testing.T) {
		ctx := context.Background()
		req := &testproto.RequestA3{Deeply: &testproto.RequestA3_SubOne{Nested: &testproto.RequestA3_SubTwo{Field: 1}}}

		_, err := call(ctx, t, req, func(w http.ResponseWriter, r *http.Request) {
			http.Redirect(w, r, "/login", http.StatusFound)
			fmt.Fprint(w, `<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 3.2 Final//EN">
<title>Redirecting...</title>
<h1>Redirecting...</h1>
<p>You should be redirected automatically to target URL: <a href="/login">/login</a>.  If not click the link.`)
		})
		if err == nil {
			t.Fatalf("expected to get error")
		}
		t.Logf("err = %v", err)
		if have, want := err.Error(), `302`; !strings.Contains(have, want) {
			t.Errorf("call.err.Error; %q != %q", have, want)
		}
		if have, want := err.Error(), `DOCTYPE HTML PUBLIC`; !strings.Contains(have, want) {
			t.Errorf("call.err.Error; %q != %q", have, want)
		}

		twerr, ok := err.(twirp.Error)
		if !ok {
			t.Fatalf("error was not a twirp.Error: type %T", err)
		}
		if have, want := twerr.Meta("prefix"), `redirected automatically`; !strings.Contains(have, want) {
			t.Errorf("call.err.Meta(\"prefix\"); %q != %q", have, want)
		}

	})
}

type testLogWriter struct {
	testing.TB
}

func (w *testLogWriter) Write(p []byte) (int, error) {
	for _, line := range bytes.Split(p, []byte("\n")) {
		w.Logf("%s", line)
	}
	return len(p), nil
}
